#ifndef SLANG_ECRA_H
#define SLANG_ECRA_H

#include <string>
#include <vector>
#include <iostream>

#include <Slang/teclado.H>

#define ANTIGO
#ifdef ANTIGO
#define Coordenada Posicao
#endif

namespace Slang {
    
    class Dimensao;

    /** Esta classe serve para representar a posio de uma clula do
        ecr. Definida no mdulo #ecra# (ficheiro de interface
        #Slang/ecra.H#). Esta classe tinha o nome #Coordenada# em verses
	anteriores da biblioteca.  Defina a macro #ANTIGO# para poder compilar 
	programas antigos. */
    class Posicao {
    public:
        /**@name Construtores */
        //@{
        /** Construtor da classe.  Por omisso a posio corresponde ao
            canto superior esquerdo do ecr.
            @param linha Nmero da linha do ecr onde se encontra a clula.
            @param coluna Nmero da coluna do ecr onde se encontra a clula.
        */
        explicit Posicao(int linha = 0, int coluna = 0);
        /** Construtor da classe por carregamento a partir de um canal.  
            @param entrada O canal de onde se carrega a posio.
            @exception ErroAoCarregar  lanada se o carregamento falhar.
            @memo Assume-se que o canal est ligado a um ficheiro cujo
            contedo foi criado usando o mtodo #guarda()#. */
        explicit Posicao(std::istream& entrada);
        //@}

        /**@name Inspectores */
        //@{
        /// Devolve a linha do ecr correspondente  posio.
        int linha() const;
        /// Devolve a coluna do ecr correspondente  posio.
        int coluna() const;
        //@}

        /**@name Modificadores */
        //@{
        /// Modifica a linha do ecr correspondente  posio.
        void linha(int linha);
        /// Modifica a coluna do ecr correspondente  posio.
        void coluna(int coluna);
        //@}

        /**@name Serializadores */
        //@{
        /** Carrega a posio a partir de um canal.  Assume-se que o canal
            est ligado a um ficheiro cujo contedo foi criado usando o mtodo
            #guarda()#.
            @param entrada O canal de onde se carrega a posio.
            @exception ErroAoCarregar  lanada se o carregamento falhar.
            Nesse caso a posio fica num estado invlido. */
        void carrega(std::istream& entrada);
        /** Guarda a posio escrevendo num canal.  O formato produzido 
            compatvel com o que o mtodo #carrega()# espera.
            @param saida O canal de onde se guarda a posio.
            @exception ErroAoGuardar  lanada se a operao falhar. */
        void guarda(std::ostream& saida) const;
        //@}

        /**@name Operadores */
        //@{
        /** Operador para soma de uma posio interpretada como um
            vector. */
        Posicao& operator += (Posicao const& c);
        /** Operador para subtraco de uma posio interpretada como um
            vector. */
        Posicao& operator -= (Posicao const& c);
        /// Operador para soma de uma dimenso.
        Posicao& operator += (Dimensao const& c);
        /// Operador para subtraco de uma dimenso.
        Posicao& operator -= (Dimensao const& c);
        //@}

    private:
        int linha_;
        int coluna_;
    };

    /**@name Operadores associados  classe #Posicao# 
       Definidos no mdulo #ecra# (ficheiro de interface
       #Slang/ecra.H#). 
     */
    //@{
    /** Soma de duas posies (o segundo operando  interpretado como um
        vector). */
    Posicao operator + (Posicao a, Posicao const& b);
    /** Diferena de duas posies (o resultado pode ser interpretado como
        um vector). */
    Posicao operator - (Posicao a, Posicao const& b);
    /** Soma de uma posio com uma dimenso. */
    Posicao operator + (Posicao a, Dimensao const& b);
    /** Subtraco de uma dimenso a uma posio. */
    Posicao operator - (Posicao a, Dimensao const& b);
    bool operator == (Posicao const& a, Posicao const& b);
    //@}

    /** Esta classe serve para representar uma dimenso no ecr.  
        tipicamente usada para representar a dimenso de caixas. Definida no
        mdulo #ecra# (ficheiro de interface #Slang/ecra.H#). */
    class Dimensao {
    public:
        /**@name Construtores */
        //@{
        /** Construtor da classe.  Por omisso a dimenso  nula.
            @param linhas Nmero da linhas do ecr ocupadas.
            @param colunas Nmero da colunas do ecr ocupadas. */
        explicit Dimensao(int linhas = 0, int colunas = 0);
        ///  possvel converter explicitamente uma posio numa dimenso.
        explicit Dimensao(Posicao const& c);
        /** Construtor da classe por carregamento a partir de um canal.  
            @param entrada O canal de onde se carrega a dimenso.
            @exception ErroAoCarregar  lanada se o carregamento falhar.
            @memo Assume-se que o canal est ligado a um ficheiro cujo
            contedo foi criado usando o mtodo #guarda()#. */
        explicit Dimensao(std::istream& entrada);
        //@}

        /**@name Inspectores */
        //@{
        /// Devolve o nmero de linhas do ecr ocupadas.
        int linhas() const;
        /// Devolve o nmero de colunas do ecr ocupadas.
        int colunas() const;
        //@}

        /**@name Modificadores */
        //@{
        /// Modifica o nmero de linhas do ecr ocupadas.
        void linhas(int linhas);
        /// Modifica o nmero de colunas do ecr ocupadas.
        void colunas(int colunas);
        //@}

        /**@name Serializadores */
        //@{
        /** Carrega a dimenso a partir de um canal.  Assume-se que o canal
            est ligado a um ficheiro cujo contedo foi criado usando o mtodo
            #guarda()#.
            @param entrada O canal de onde se carrega a dimenso.
            @exception ErroAoCarregar  lanada se o carregamento falhar.
            Nesse caso a dimenso fica num estado invlido. */
        void carrega(std::istream& entrada);
        /** Guarda a dimenso escrevendo num canal.  O formato produzido 
            compatvel com o que o mtodo #carrega()# espera.
            @param saida O canal de onde se guarda a dimenso.
            @exception ErroAoGuardar  lanada se a operao falhar. */
        void guarda(std::ostream& saida) const;
        //@}

        /**@name Operadores */
        //@{
        /** Operador para soma de uma dimenso. */
        Dimensao& operator += (Dimensao const& c);
        /** Operador para subtraco de uma dimenso. */
        Dimensao& operator -= (Dimensao const& c);
        //@}

    private:
        int linhas_;            // -1 significa to alto quanto possvel.
        int colunas_;           // -1 significa to largo quanto possvel.
    };

    /**@name Operadores associados  classe #Dimensao#
       Definidos no mdulo #ecra# (ficheiro de interface
       #Slang/ecra.H#). */
    //@{
    /// Soma de duas dimenses.
    Dimensao operator + (Dimensao a, Dimensao const& b);
    /// Subtraco de duas dimenses.
    Dimensao operator - (Dimensao a, Dimensao const& b);
    //@}

    /** Esta classe serve para representar caixas no ecr.  Uma caixa  um
        conjunto de clulas formando um rectngulo de lados no
        oblquos. Definida no mdulo #ecra# (ficheiro de interface
        #Slang/ecra.H#). */
    class Caixa {
    public:
        /**@name Construtores */
        //@{
        /** Construtor da classe.  Por omisso a caixa est vazia.
            @param origem Posicao da clula no canto superior esquerdo da
            caixa. 
            @param dimensao Dimenso da caixa. */
        explicit Caixa(Posicao const& origem = Posicao(), 
                       Dimensao const& dimensao = Dimensao());
        /** Construtor da classe.
            @param origem Posicao da clula no canto superior esquerdo da
            caixa. 
            @param destino Posicao da clula no canto inferior direito da
            caixa. */
        explicit Caixa(Posicao const& a, Posicao const& b);
        /** Construtor da classe por carregamento a partir de um canal.  
            @param entrada O canal de onde se carrega a caixa.
            @exception ErroAoCarregar  lanada se o carregamento falhar.
            @memo Assume-se que o canal est ligado a um ficheiro cujo
            contedo foi criado usando o mtodo #guarda()#. */
        explicit Caixa(std::istream& entrada);
        //@}
        
        /**@name Inspectores */
        //@{
        /// Devolve a posio da clula no canto superior esquerdo da caixa.
        Posicao origem() const;
        /// Devolve a posio da clula no canto inferior direito da caixa.
        Posicao destino() const;
        /// Devolve a dimenso da caixa.
        Dimensao dimensao() const;
        //@}

        /** Devolve #true# se a caixa estiver vazia, ou seja, se tiver
            dimenso nula. */
        bool vazia() const;

        /**@name Relaes geomtricas */
        //@{
        /// Devolve #true# se a posio #p# estiver sobre a caixa (na borda
        // ou no seu interior).
        bool sobre(Posicao const& p) const;
        /// Devolve #true# se a posio #p# estiver sobre a borda da caixa.
        bool sobreABorda(Posicao const& p) const;
        //@}

        /**@name Modificadores */
        //@{
        /** Modifica a posio do canto superior esquerdo da caixa (mas a
            sua dimenso mantm-se!). */
        void origem(Posicao const& origem);
        /** Modifica a posio do canto inferior direito da caixa (no
            modifica a posio do canto superior esquerdo, pelo que a
            sua dimenso altera-se!). */
        void destino(Posicao const& destino);
        /// Modifica a dimenso da caixa.
        void dimensao(Dimensao const& dimensao);
        //@}

        /**@name Operadores */
        //@{
        /// Desloca a caixa de #c#, que  interpretado como um vector.
        Caixa& operator += (Posicao const& c);
        /// "Soma" com uma caixa (resulta a caixa envolvente da unio).
        Caixa& operator += (Caixa const& c);
        /// "Multiplica" por uma caixa (resulta a caixa de interseco).
        Caixa& operator *= (Caixa const& c);
        //@}

        /**@name Serializadores */
        //@{
        /** Carrega a caixa a partir de um canal.  Assume-se que o canal
            est ligado a um ficheiro cujo contedo foi criado usando o mtodo
            #guarda()#.
            @param entrada O canal de onde se carrega a caixa.
            @exception ErroAoCarregar  lanada se o carregamento falhar.
            Nesse caso a caixa fica num estado invlido. */
        void carrega(std::istream& entrada);
        /** Guarda a caixa escrevendo num canal.  O formato produzido 
            compatvel com o que o mtodo #carrega()# espera.
            @param saida O canal de onde se guarda a caixa.
            @exception ErroAoGuardar  lanada se a operao falhar. */
        void guarda(std::ostream& saida) const;
        //@}

    private:
        Posicao origem_;
        Dimensao dimensao_;
    };

    /**@name Operadores associados  classe #Caixa# 
       Definidos no mdulo #ecra# (ficheiro de interface
       #Slang/ecra.H#). */
    //@{
    /// Soma de caixa posio: devolve caixa deslocada.
    Caixa operator + (Caixa a, Posicao const& c);
    /// Soma de caixa com posio: devolve caixa deslocada.
    Caixa operator + (Posicao const& c, Caixa a);
    /** "Soma" de duas caixas: devolve caixa envolvente das duas caixas. */
    Caixa operator + (Caixa a, Caixa const& b);
    /** "Produto" de duas caixas: devolve interseco das duas caixas. */
    Caixa operator * (Caixa a, Caixa const& b);// caixa interseco.
    //@}

    /**@name Constantes associadas  classe #Caixa#
       Definidas no mdulo #ecra# (ficheiro de interface
       #Slang/ecra.H#). */
    //@{
    /// Constante representando uma caixa vazia.
    Caixa const caixa_vazia = Caixa();
    /// Constante representando a caixa universal (enche todo o espao).
    Caixa const caixa_universal = Caixa(Posicao(-10000, -10000), 
                                        Dimensao(10000, 10000));
    // Afazer: pr numric limits quando existir em Linux.
    //@}


    /** Este tipo enumerado representa as possveis cores do texto e do fundo
        de cada clula do ecr. Note-se que nem todas as cores so vlidas
        para o fundo em todos os terminais!  Experimente para verificar quais
         que funcionam bem... Ver na documentao abaixo os nomes das cores
        usados para insero e extrao de canais.  Definido no mdulo #ecra#
        (ficheiro de interface #Slang/ecra.H#). */
    enum Cor {
        /// Nome num canal: #preto#.
        preto,
        primeira_cor = preto,
        /// Nome num canal: #cinza#.
        cinza,
        /// Nome num canal: #vermelho#.
        vermelho,
        /// Nome num canal: #vermelho-brilhante#.
        vermelho_brilhante,
        /// Nome num canal: #verde#.
        verde,
        /// Nome num canal: #verde-brilhante#.
        verde_brilhante,
        /// Nome num canal: #castanho#.
        castanho,
        /// Nome num canal: #amarelo#.
        amarelo,
        /// Nome num canal: #azul#.
        azul,
        /// Nome num canal: #azul-brilhante#.
        azul_brilhante,
        /// Nome num canal: #magenta#.
        magenta,
        /// Nome num canal: #magenta-brilhante#.
        magenta_brilhante,
        /// Nome num canal: #ciano#.
        ciano,
        /// Nome num canal: #ciano-brilhante#.
        ciano_brilhante,
        /// Nome num canal: #cinzento-claro#.
        cinzento_claro,
        /// Nome num canal: #branco#.
        branco,
        ultima_cor = branco
    };

    /**@name Operadores associados ao tipo enumerado #Cor#
       Definidos no mdulo #ecra# (ficheiro de interface
       #Slang/ecra.H#). */
    //@{
    /// Operador de insero de cores num canal.
    std::ostream& operator << (std::ostream& saida, Cor cor);
    /// Operador de extraco de cores de um canal.
    std::istream& operator >> (std::istream& entrada, Cor& cor);
    //@}

    /** Constante que guarda o nmero total de cores.  
        Definida no mdulo #ecra# (ficheiro de interface #Slang/ecra.H#). */
    int const numero_de_cores = ultima_cor + 1;


    /** Esta classe  um solito: a sua nica instncia representa o ecr.
        Note-se que o ecr  virtual.  As alteraes nele feitas s tm efeito
        no ecr real depois de uma operao de refrescamento! Definida no
        mdulo #ecra# (ficheiro de interface #Slang/ecra.H#). 

        O ecr permite vrias operaes: 
        \begin{itemize}
        \item inspeccionar a dimenso do ecr e a posio do cursor;
        \item verificar se o ecr foi redimensionado;
        \item modificar a posio do cursor (de vrias formas diferentes);
        \item copiar ({\em copy}) ou colar ({\em paste}) um troo de/para uma
        posio do ecr;
        \item apagar vrias partes do ecr;
        \item refrescar o ecr;
        \item desenhar uma caixa no ecr;
        \item enviar para o ecr um caractere, um inteiro, uma cadeia de
        caracteres, uma caixa, um troo ou um dos manipuladores definidos;
        \item etc.
        \end{itemize}

        {\bf Exemplo}

        \begin{verbatim}
        // Cria uma caixa de dimenso 1010 com origem na posio (0,0): 
        Caixa caixa(Posicao(0, 0), Dimensao(10, 10));  
        
        // Copia o troo de ecr que se encontra sob a caixa:
        Ecra::Troco troco = ecra.copia(caixa); 

        // Cria um novo ObjectoCor:
        Ecra::ObjectoCor cor(vermelho, verde);
        // Move o cursor para a posio (30,30), cola o troo copiado acima
        // nessa posio, muda a cor activa e refresca o ecr de modo a que
        // estas modificaes se tornem visveis:
        ecra << cursor(30, 30) << troco << cor  << refresca;
        \end{verbatim}

        {\bf Ateno}

        As alteraes s so visveis depois de chamada uma das funes de
        refrescamento do ecr. Quando h um redimensionamento do ecr deve
        ser chamado o procedimento #refrescaTudo()#. Quando so feitas pequenas
        alteraes deve ser chamado o procedimento #refresca()#.  Tambm podem
        ser usados os manipuladores #refresca# e #refresca_tudo# para obter o
        mesmo efeito.

        {\bf Exemplo de utilizao}

        Este programa escreve no ecr todas as letras do alfabeto (maisculas
        e minsculas) e depois, a cada presso de uma tecla, escreve os
        nmeros de zero a nove com outra cor e com largura proporcional ao
        nmero escrito.

        \begin{verbatim}
        #include <Slang/slang.H> 
        
        using namespace Slang; 
        
        extern "C" {
        #include <unistd.h>  // para sleep()
        }
        
        int main () 
        { 
            for(char c = 'a'; c != 'z' + 1; ++c) { 
                ecra << c << refresca; 
                sleep(1);
            } 
        
            teclado.tecla(); 
        
            Ecra::ObjectoCor cor(amarelo, azul); 
        
            ecra << cor << cursor(0, 0); 
            for (int c = 0; c != 10; ++c) { 
                ecra.baixo(); 
                ecra << parado << largura(c) << char('0' + c) << refresca; 
                teclado.tecla();
            } 
        } 
        \end{verbatim}
    */
    class Ecra {
    public:

        // Vrias classes embutidas para definir manipuladores.  Muitos
        // mtodos podem ser invocados indirectamente usando manipuladores:
        struct Fundo;
        struct Parado;
        struct Largura;
        struct Refresca;
        struct RefrescaTudo;
        struct Apaga;
        struct ApagaFimDaLinha;
        struct ApagaFimDoEcra;
        struct Campainha;

        /** Esta classe  usada para indicar a cor das clulas do ecr (as
            clulas tm uma cor de texto e outra de fundo).  O sistema usado
            pela biblioteca C slang (base desta biblioteca C++ Slang++) 
            complicado pelo facto de se basear no conceito de palete.  A ideia
             que se deve usar um objecto cor para reservar uma posio da
            palete.  Todas as clulas escritas usando este objecto cor
            guardaro na realidade uma referncia para o respectivo objecto
            cor.  Assim, se se alterar a cor do texto ou do fundo desse
            objecto cor, mudar a cor de todas as respectivas clulas!
            Note-se que, devido a limitaes da biblioteca de base que est a
            ser usada (C slang), nmero de objectos cor definidos em cada
            momento no dever ultrapassar os 255!  Se isso acontecer as cores
            deixam de poder ser alteradas independentemente, com resultados
            muito estranhos...  Note-se ainda que nem todas as cores so
            utilizveis como cor de fundo em todas as consolas.

	    Os objectos cor no so copiveis!  Este facto deve-se a que cada
	    objecto criado ocupa um posio da palete.  Assim, no  possvel
	    passar um objecto cor como argumento para uma funo ou
	    procedimento, excepto por referncia.

            O ecr tem sempre uma cor activa (nem que seja a cor do fundo do
            ecr que por omisso  de texto branco sobre fundo preto).

            Pode-se mudar a cor activa de vrias maneiras, por exemplo usando
            o operador #Ecra& operator << (ObjectoCor const& objecto_cor)# do
            seguinte modo:

            \begin{verbatim}
            Ecra::ObjectoCor cor(vermelho / * texto * /, verde / * fundo * /); 
            ecra << cor;
            \end{verbatim}

            A partir desta instruo tudo que for escrito passar a ser em
            vermelho sobre verde at que a cor seja mudada de novo.  Note-se
            que  recomendvel que a varivel #cor# pressista enquanto
            existirem no ecr clulas desenhadas com recurso a essa cor.

            No entanto, aps uma colagem de um troo de ecr copiado
            previamente a cor activa volta a ser a do fundo do ecr.

            A escolha das cores utilizadas num programa deve ter sempre como
            objectivo tornar to clara quanto possvel a utilizao do
            programa. No se deve usar todas as cores disponveis apenas
            "porque esto l".  Deve-se tambm tambm ter o cuidado de
            escolher cores de fundo e de texto que sejam minimamente
            contrastantes de modo a que o utilizador perceba bem o que est no
            ecr. */
        // Esta classe s est definida aqui dentro por causa do doc++, que se 
        // d mal com ela definida fora da classe Ecra...
        class ObjectoCor {
        public:
            /** Construtor da classe.
                @param texto Cor inicial do texto.
                @param fundo Cor inicial do fundo. */
            ObjectoCor(Cor texto, Cor fundo);
        
            // Destrutor:
            ~ObjectoCor();

            /**@name Inspectores */
            //@{
            /// Devolve a cor do texto corrente deste objecto.
            Cor texto() const;
            /// Devolve a cor do fundo corrente deste objecto.
            Cor fundo() const;
            //@}

            /**@name Modificadores */
            //@{
            /** Modifica a cor do texto de todas as clulas do ecr desenhadas 
                usando este objecto. */
            void texto(Cor texto);
            /** Modifica a cor do fundo de todas as clulas do ecr desenhadas 
                usando este objecto. */
            void fundo(Cor fundo);
            //@}

        private:
            // No se podem copiar estes objectos!
            ObjectoCor(ObjectoCor const&);
            ObjectoCor& operator = (ObjectoCor const&);
            Cor texto_;         // cor do texto.
            Cor fundo_;         // cor do fundo.
            int objecto;                // nmero do objecto (entrada na palete).

            // A classe Ecra precisa de aceder ao nmero do objecto:
            friend Ecra;
        };

        /** Esta classe serve para representar um troo de ecr.  Os troos de 
            ecr so teis para guardar partes do ecr, que mais tarde podem
            ser desenhados onde se entender. 

            Uma varivel do tipo #Troco# (leia-se #Troo#) guarda uma parte do
            ecr com todas as suas propriedades. Esta classe  especialmente
            til em operaes como cortar e colar ({\em copy & paste}) para
            guardar a parte do ecr que se pretende reproduzir e depois pr
            este mesmo troo de ecr noutra ou na mesma posio.  */
        // Esta classe s est definida aqui dentro por causa do doc++, que se 
        // d mal com ela definida fora da classe Ecra...
        class Troco {
        public:
            /** Construtor da classe.   pouco usual ser usado directamente.
                Normalmente  a prpria classe ecr que constri o troo e o
                devolve (e portanto o construtor mais usado  o construtor por 
                cpia).
                @param dimensao Dimenso do troo de ecr a guardar. */
            explicit Troco(Dimensao const& dimensao);

            /// Devolve a dimenso do troo.
            Dimensao dimensao() const;

        private:
            Dimensao dimensao_;
            vector<unsigned short> dados;
            // A classe Ecra precisa de acesso a dados privados desta classe.
            friend Ecra;
        };

        /** Construtor da classe.  Tal como a biblioteca se encontra definida,
            este construtor no pode ser usado directamente!  No entanto, numa
            verso futura, onde no exista varivel global para representar o
            ecr, o programador utilizador pode vir a us-lo.  Encarrega-se de
            inicializar o "screen management" Slang.  Se se construir com
            #limita_cursor# #true#, o ecr garantir que o cursor se encontra
            dentro dos seus limites em cada instante.  Por omisso essa
            verificao no  feita.

            @param texto Cor do texto das chamadas clulas de fundo, para as
            quais no foi indicado qualquer objecto cor (cor por omisso).
            @param fundo Cor do fundo das chamadas clulas de fundo, para as
            quais no foi indicado qualquer objecto cor (cor por omisso).
            @param limita_cursor Indica se a posio do cursor deve ser
            limitada  dimenso do ecr.
        */
        explicit Ecra(Cor texto = branco, Cor fundo = preto, 
                      bool limita_cursor = false);

        /** Destrutor da classe.  Encarrega-se de finalizar o "screen
            management" Slang. */
        ~Ecra();

        /**@name Inspectores */
        //@{
        /// Devolve a dimenso corrente do ecr.
        Dimensao dimensao() const;
        /// Devolve a posio do cursor.
        Posicao cursor() const;
        /** Indica se o ecr foi redimensionado ou no.  Isto 
            particularmente relevante se o programa estiver a correr numa
            consola dentro de um ambiente de janelas.  Nessas circunstncias o 
            utilizador pode alterar a dimenso da janela, sendo com isso
            alterada a dimenso do ecr.  Se isso acontecer o programa dever
            redesenhar e refrescar o ecr. */
        bool redimensionado() const;
        //@}

        /**@name Modificadores */
        //@{
        /** Modifica a cor por omisso, i.e., a cor das clulas que constituem
            o fundo do ecr. 
            @param texto Nova cor do texto para as clulas do fundo do ecr.
            @param fundo Nova cor do fundo para as clulas do fundo do ecr. */
        void atributos(Cor texto, Cor fundo);
        /** Modifica a posio do cursor.
            @param posicao Nova posio do cursor: limitada ou no  dimenso
            do ecr consoante o modo de construo do ecr (por omisso no h 
            limitao). */
        void cursor(Posicao const& posicao);
        /// Desloca cursor uma linha para cima.
        void cima();
        /// Desloca cursor uma linha para baixo.
        void baixo();
        /// Desloca cursor uma coluna para a esquerda.
        void esquerda();
        /// Desloca cursor uma coluna para a direita.
        void direita();
        /** Desloca cursor na direco indicada pela tecla. 
            @param tecla A tecla.  Se for uma seta, o cursor  deslocado.  Se
            no for, o cursor mantm-se. */
        void desloca(Tecla const& tecla);
        //@}

        /**@name Mtodos para cpia e cola de troos de ecr */
        //@{
        /** Devolve uma cpia de um troo do ecr. 
            @param caixa O troo do ecr  o que se encontra sob esta caixa. */
        Troco copia(Caixa const& caixa) const;
        /// Devolve uma cpia do ecr inteiro.
        Troco copia() const;
        /** Devolve uma cpia do troo de ecr com um dada dimenso e
            com origem na posio corrente do cursor. 
            @param dimensao A dimenso do troo. */
        Troco copia(Dimensao const& dimensao) const;
        /** Cola no ecr, na posio indicada, um troo de ecr.  O cursor 
            deslocado para a posio imediatamente aps a ltima coluna da
            ltima linha onde o troo foi colocado, a no ser que se tenha
            pedido para manter o cursor parado.  Ateno!  O objecto cor que
            estava a ser usado antes da colagem  perdido!  As prximas
            escritas so feitas usando a cor do fundo.
            @param troco Troo a colar.
            @param origem Posio onde o troo deve ser colado (por omisso
            cola no canto superior esquerdo do ecr). 
            @see parado */
        void cola(Troco const& troco, Posicao const& origem = Posicao());
        //@}

        /**@name Mtodos para desenhar no ecr */
        //@{
        /// Toca a "campanha" (normalmente o ecr pisca).
        void campainha() const;
        /** Apaga (i.e., pinta com cor do fundo) at ao fim da linha do
            cursor. */
        void apagaFimDaLinha(); 
        /** Apaga (i.e., pinta com cor do fundo) at ao fim do ecr a partir
            do cursor. */
        void apagaFimDoEcra();
        /// Apaga (i.e., pinta com cor do fundo) todo o ecr.
        void apaga();
        /** Refresca o ecr, ou seja, reflecte no ecr real as ltimas
            alteraes realizadas no ecr virtual. */
        void refresca() const;
        /** Refresca totalmente o ecr, ou seja, reflecte no ecr real todo o
            ecr virtual. */
        void refrescaTudo() const;
        /** Desenha uma caixa no ecr, sendo a borda desenhada com caracteres
            especiais. 
            @param caixa Caixa a desenhar.
        */
        void desenhaCaixa(Caixa const& caixa);
        //@}

        /**@name Operadores de insero no ecr de tipos usuais */
        //@{
        /** Insere um caracter na posio do cursor.  O cursor 
            deslocado uma posio para a direita, a no ser que se tenha
            pedido para manter o cursor parado.  Se se tiver especificado uma
            largura antes, so acrescentados espaos at perfazer a largura
            indicada.
            @param c O caractere a inserir.
            @see parado
            @see largura */
        Ecra& operator << (char c);
        /** Insere um nmero inteiro na base 10 na posio do cursor.  O
            cursor  deslocado para a posio  direita do ltimo dgito
            escrito, a no ser que se tenha pedido para manter o cursor
            parado.  Se se tiver especificado uma
            largura antes, so acrescentados espaos at perfazer a largura
            indicada.  Se a largura for insuficiente para representar o
            inteiro, s so mostrados os primeiros dgitos.
            @param i O inteiro a inserir.
            @see parado 
            @see largura */
        Ecra& operator << (int i);
        /** Insere uma cadeia de caracteres na posio do cursor.  O
            cursor  deslocado para a posio  direita do ltimo caractere
            escrito, a no ser que se tenha pedido para manter o cursor
            parado.  Se se tiver especificado uma
            largura antes, so acrescentados espaos at perfazer a largura
            indicada.  Se a cadeia for maior que a largura especificada, s
            so mostrados os primeiros caracteres.
            @param c A cadeia a inserir.
            @see parado 
            @see largura */
        Ecra& operator << (string const& c);
        // Afazer: template para escrita de tipos no previstos usando
        // stringstream.
        //@}

        /**@name Operadores de insero de objectos especiais no ecr */
        //@{
        /** Desenha uma caixa no ecr, sendo a borda desenhada com caracteres
            especiais. 
            @param caixa Caixa a desenhar.
            @see desenhaCaixa
        */
        Ecra& operator << (Caixa const& caixa);
        /** Cola no ecr, na posio do cursor, um troo de ecr.  O cursor 
            deslocado para a posio imediatamente aps a ltima coluna da
            ltima linha onde o troo foi colocado, a no ser que se tenha
            pedido para manter o cursor parado.
            @param troco Troo a colar.
            @see parado
            @see cola */
        Ecra& operator << (Troco const& troco);
        /** Altera a posio do cursor para a posio dada.
            @param posicao Nova posio do cursor.
            @see cursor */
        Ecra& operator << (Posicao const& posicao);
        /** Altera o objecto cor das prximas operaes de escrita no ecr
            (com excepo das colagens!).
            @param objecto_cor Objecto cor a usar nas prximas escritas no
            ecr. 
            @see cola */
        Ecra& operator << (ObjectoCor const& objecto_cor);
        //@}

        /** Operadores de extraco de um troo do ecr.  Extrai um troo a
            partir da posio corrente do cursor.
            @param troco Troo a extrair (copiar). */
        Ecra const& operator >> (Troco& troco) const;
        
        // Operadores de insero de manipuladores no ecr:
        Ecra& operator << (Fundo const&);
        Ecra& operator << (Parado const&);
        Ecra& operator << (Largura const&);
        Ecra& operator << (Refresca const&);
        Ecra& operator << (RefrescaTudo const&);
        Ecra& operator << (Apaga const&);
        Ecra& operator << (ApagaFimDaLinha const&);
        Ecra& operator << (ApagaFimDoEcra const&);
        Ecra& operator << (Campainha const&);

    private:
        // Mtodos auxiliares.  Usados para a implementao:
        
        // Colocar atributos num dado objecto:
        void atributos(Cor texto, Cor fundo, int objecto);

        // Verificar se o tamanho do ecr sofreu alteraes:
        void verificaTamanho() const;

        // Assim ningum pode invocar o construtor por cpia:
        Ecra(Ecra const&) {}

        // S os membros da classe podem posicionar o cursor sem que isso seja 
        // considerado uma alterao.  Quando o fizerem tm de se
        // responsabilizar por voltar a por o cursor na posio original.
        void cursor(Posicao const& posicao) const;

        // Usados pelos objectos cor para reservar ou libertar um lugar na
        // palete:
        int reservaObjecto();
        void libertaObjecto(int objecto);

        // Variveis membro:
        bool limita_cursor;     // o cursor est a ser limitado?
        mutable bool redimensionado_; // o ecr foi redimensionado?
        bool para_no_proximo; // a prxima operao de escrita simples no
                                // altera a posio do cursor.
        int largura_do_proximo; // a prxima operao de escrita simples
                                // ocupar pelo menos os espaos indicados.
        vector<int> ocupacao_objecto;// estado de ocupao dos objectos da
                                // palete.


        // Variveis membro de classe:
        static bool existe_instancia;// Para no deixar existir qualquer outra 
                                // instncia do ecr.
        static volatile bool tamanho_mudou; // o ecr foi redimensionado?
                                // alterada pelo menipulador do sinal SIGWINCH.
        static char *nome_de_cor[numero_de_cores];// nomes das cores usados
                                // pelo Slang.

        // Mtodos de classe:
        static void manipuladorTamanhoMudou(int sinal);// manipulador do sinal
                                // SIGWINCH.

        // Amizades:
        friend ObjectoCor;
    };


    /** Uma varivel global representando o ecr.  A ideia  que substitua a
        varivel global #cout# usada normalmente para escrever no ecr.
        Definida no mdulo #ecra# (ficheiro de interface #Slang/ecra.H#). */
    extern Ecra ecra;


    // Classes para os manipuladores:

    struct Ecra::Fundo {
    };

    struct Ecra::Parado {
    };
    
    struct Ecra::Largura {
        explicit Largura(int largura = 0); 
        Largura operator() (int largura) const;
        int largura;
    };
    
    struct Ecra::Refresca {
    };

    struct Ecra::RefrescaTudo {
    };

    struct Ecra::ApagaFimDaLinha {
    };
    
    struct Ecra::ApagaFimDoEcra {
    };
    
    struct Ecra::Apaga {
    };

    struct Ecra::Campainha {
    };


    /**@name Manipuladores do ecr
       Definidos no mdulo #ecra# (ficheiro de interface
       #Slang/ecra.H#). */
    //@{
    /**@name refresca
       Manipulador usado para refrescar o ecr.  Usado como
       se segue:
       \begin{verbatim}
       // Coloca cursor no canto superior esquerdo do ecr:
       ecra << cursor(0, 0);
       // Escreve "Ol mundo!" a partir dessa posio:
       ecra << "Ol mundo";
       // Refresca o ecr:
       ecra << refresca;
       \end{verbatim} 
       @see Ecra::refresca */
    /**@name refresca_tudo
       Manipulador usado para fazer o refrescamento total do ecr.  Usado como 
       se segue:
       \begin{verbatim}
       ecra << refresca_tudo;
       \end{verbatim} 
       @see Ecra::refrescaTudo */
    /**@name cursor
       Manipulador usado para colocar o cursor numa dada posio.  Usar como
       se segue:
       \begin{verbatim}
       // Coloca cursor na linha 10 coluna 20 do ecr:
       ecra << cursor(10, 20);
       // Escreve "Ol mundo!" a partir dessa posio:
       ecra << "Ol mundo";
       // Refresca o ecr:
       ecra << refresca;
       \end{verbatim} 
       @see Ecra::cursor */
    /**@name parado
       Manipulador usado para manter o cursor na posio corrente depois da
       {\bf prxima} operao de escrita no ecr.  Usar como se segue:
       \begin{verbatim}
       // Coloca cursor no canto superior esquerdo do ecr:
       ecra << cursor(0, 0);
       // Escreve "Ol mundo!" mantendo a posio do cursor no topo superior
       // esquerdo do ecr:
       ecra << parado << "Ol mundo";
       // Refresca o ecr:
       ecra << refresca;
       \end{verbatim} */
    /**@name largura
       Manipulador usado para indicar o nmero de caracteres a escrever na
       {\bf prxima} operao de escrita de tipos usuais (#char#, #int#,
       #std::string#).  Usar como se segue:
       \begin{verbatim}
       // Coloca cursor no canto superior esquerdo do ecr:
       ecra << cursor(0, 0);
       // Define objecto cor para a prxima escrita:
       Ecra::ObjectoCor cor(amarelo, vermelho);
       // Escreve "Ol mundo! " usando a cor indicada e ocupando 30 caracteres:
       ecra << oor << largura(30) << "Ol mundo!";
       \end{verbatim} */
    /**@name fundo
       Manipulador usado para indicar que as prximas escritas devem ser
       realizadas usando a cor do fundo do ecr.  Usar como se segue:
       \begin{verbatim}
       // Coloca cursor no canto superior esquerdo do ecr:
       ecra << cursor(0, 0);
       // Define objecto cor para a prxima escrita:
       Ecra::ObjectoCor cor(amarelo, vermelho);
       // Escreve "Ol mundo! " usando a cor indicada:
       ecra << oor << "Ol mundo! ";
       // Escreve "Ol mundo!" mas usando a cor do fundo:
       ecra << fundo << "Ol mundo!";
       // Refresca o ecr:
       ecra << refresca;
       \end{verbatim} */
    /**@name caixa
       Manipulador usado para desenhar uma caixa com caracteres especiais numa
       dada posio.  Usar como se segue:
       \begin{verbatim}
       // Desenha caixa cor origem na linha 10 coluna 20 com 5 linhas e 15
       // colunas:
       ecra << caixa(10, 20, 5, 15);
       // Refresca o ecr:
       ecra << refresca;
       \end{verbatim} 
       @see Ecra::desenhaCaixa */
    /**@name apaga
       Manipulador usado para apagar o ecr (ou seja, para o pintar com a cor
       do fundo).  Usar como se segue:
       \begin{verbatim}
       ecra << apaga;
       // Refresca o ecr:
       ecra << refresca;
       \end{verbatim} 
       @see Ecra::apaga */
    /**@name apaga_fim_da_linha
       Manipulador usado para apagar at ao fim da linha do cursor (ou seja,
       para o pintar com a cor do fundo).  Usar como se segue:
       \begin{verbatim}
       // Apaga linha 10 a partir da coluna 20:
       ecra cursor(10, 20) << apaga_fim_da_linha;
       // Refresca o ecr:
       ecra << refresca;
       \end{verbatim} 
       @see Ecra::apagaFimDaLinha */
    /**@name apaga_fim_do_ecra
       Manipulador usado para apagar desde a posio do cursor at ao fim do
       ecr (ou seja, para o pintar com a cor do fundo).  Usar como se segue:
       \begin{verbatim}
       // Apaga ecr a partir da coluna 20 da linha 10:
       ecra cursor(10, 20) << apaga_fim_do_ecra;
       // Refresca o ecr:
       ecra << refresca;
       \end{verbatim} 
       @see Ecra::apagaFimDoEcra */
    /**@name campainha
       Manipulador usado para fazer soar a campainha.  Usar como se segue:
       \begin{verbatim}
       ecra << campainha;
       \end{verbatim} 
       @see Ecra::campainha */
    //@}


    // Declarao de funes para manipuladores:

    Posicao cursor(Posicao const& posicao);
    Posicao cursor(int linha, int coluna);
    Caixa caixa(Posicao const& origem = Posicao(), 
                Dimensao const& dimensao = Dimensao());
    Caixa caixa(int linha, int coluna, int linhas, int colunas);
    

    // Declarao de constantes globais:
    
    Ecra::Fundo const fundo = Ecra::Fundo();
    Ecra::Parado const parado = Ecra::Parado();
    Ecra::Largura const largura;
    Ecra::Refresca const refresca = Ecra::Refresca();
    Ecra::RefrescaTudo const refresca_tudo = Ecra::RefrescaTudo();
    Ecra::Apaga const apaga = Ecra::Apaga();
    Ecra::ApagaFimDaLinha const apaga_fim_da_linha = Ecra::ApagaFimDaLinha();
    Ecra::ApagaFimDoEcra const apaga_fim_do_ecra = Ecra::ApagaFimDoEcra();
    Ecra::Campainha const campainha = Ecra::Campainha();

    /** Constante global com os nomes das cores.  Indexvel com os valores do
        tipo enumerado #Cor#.  Definida no mdulo #ecra# (ficheiro de
        interface #Slang/ecra.H#). */
    extern string const nomes_das_cores[numero_de_cores];

}

#include <Slang/ecra_impl.H>

#endif // SLANG_ECRA_H
