#include <cassert>
#include <cstdio>

extern "C" {
#include <slang/slang.h>
#include <signal.h>
}

// Mtodos inline da classe Posicao:

// Construtores:
inline Slang::Posicao::Posicao(int linha, int coluna)
    : linha_(linha), coluna_(coluna) {
}

inline Slang::Posicao::Posicao(istream& entrada) {
    carrega(entrada);
}

// Inspectores:
inline int Slang::Posicao::linha() const {
    return linha_;
}

inline int Slang::Posicao::coluna() const {
    return coluna_;
}

// Modificadores:
inline void Slang::Posicao::linha(int linha) {
    linha_ = linha;
}

inline void Slang::Posicao::coluna(int coluna) {
    coluna_ = coluna;
}

inline Slang::Posicao& 
Slang::Posicao::operator += (Posicao const& c) {
    linha_ += c.linha_;
    coluna_ += c.coluna_;
    return *this;
}

inline Slang::Posicao& 
Slang::Posicao::operator -= (Posicao const& c) {
    linha_ -= c.linha_;
    coluna_ -= c.coluna_;
    return *this;
}

inline Slang::Posicao& 
Slang::Posicao::operator += (Dimensao const& c) {
    linha_ += c.linhas();
    coluna_ += c.colunas();
    return *this;
}

inline Slang::Posicao& 
Slang::Posicao::operator -= (Dimensao const& c) {
    linha_ -= c.linhas();
    coluna_ -= c.colunas();
    return *this;
}

// Operadores associados:

inline Slang::Posicao 
Slang::operator + (Slang::Posicao a, Slang::Posicao const& b) {
    return a += b;
}

inline Slang::Posicao 
Slang::operator - (Slang::Posicao a, Slang::Posicao const& b) {
    return a -= b;
}

inline Slang::Posicao 
Slang::operator + (Slang::Posicao a, Slang::Dimensao const& b) {
    return a += b;
}

inline Slang::Posicao 
Slang::operator - (Slang::Posicao a, Slang::Dimensao const& b) {
    return a -= b;
}

inline bool 
Slang::operator == (Slang::Posicao const& a, Slang::Posicao const& b) {
    return a.linha() == b.linha() && a.coluna() == b.coluna();
}


// Mtodos inline da classe Dimensao:

inline Slang::Dimensao::Dimensao(int l, int c) {
    linhas(l);
    colunas(c);
}

inline Slang::Dimensao::Dimensao(Posicao const& c) {
    linhas(c.linha());
    colunas(c.coluna());
}
      
inline Slang::Dimensao::Dimensao(istream& entrada) {
    carrega(entrada);
}
      
// Inspectores:
inline int Slang::Dimensao::linhas() const {
    return linhas_;
}

inline int Slang::Dimensao::colunas() const {
    return colunas_;
}

// Modificadores:
inline void Slang::Dimensao::linhas(int linhas) {
    linhas_ = linhas < 0 ? 0 : linhas;
}

inline void Slang::Dimensao::colunas(int colunas) {
    colunas_ = colunas < 0 ? 0 : colunas;
}

// Operadores:
inline Slang::Dimensao& Slang::Dimensao::operator += (Dimensao const& c) {
    linhas(linhas() + c.linhas());
    colunas(colunas() + c.colunas());
    return *this;
}

inline Slang::Dimensao& Slang::Dimensao::operator -= (Dimensao const& c) {
    linhas(linhas() - c.linhas());
    colunas(colunas() - c.colunas());
    return *this;
}

// Operadores associados:
inline Slang::Dimensao Slang::operator + (Dimensao a, Dimensao const& b) {
    return a += b;
}

inline Slang::Dimensao Slang::operator - (Dimensao a, Dimensao const& b) {
    return a -= b;
}

// Mtodos inline da classe Caixa:

// Construtores:
inline Slang::Caixa::Caixa(Posicao const& origem, Dimensao const& dimensao)
    : origem_(origem), dimensao_(dimensao) {
}

inline Slang::Caixa::Caixa(Posicao const& a, Posicao const& b)
    : origem_(min(a.linha(), b.linha()), min(a.coluna(), b.coluna())), 
      dimensao_(Posicao(max(a.linha(), b.linha()) + 1, 
			   max(a.coluna(), b.coluna()) + 1) - origem_) {
}

inline Slang::Caixa::Caixa(istream& entrada)
    : origem_(entrada), dimensao_(entrada) {
}


// Inspectores:
inline Slang::Posicao Slang::Caixa::origem() const {
    return origem_;
}

inline Slang::Posicao Slang::Caixa::destino() const {
    return origem_ + dimensao_ - Posicao(1, 1);
}

inline Slang::Dimensao Slang::Caixa::dimensao() const {
    return dimensao_;
}

inline bool Slang::Caixa::vazia() const {
    return dimensao().linhas() == 0 || dimensao().colunas() == 0;
}

inline bool Slang::Caixa::sobre(Posicao const& c) const {
    int const linha = origem_.linha();
    int const coluna = origem_.coluna();
    int const linha_final = linha + dimensao_.linhas();
    int const coluna_final = coluna + dimensao_.colunas();
    return c.linha() >= linha && c.linha() < linha_final &&
	c.coluna() >= coluna && c.coluna() < coluna_final;
}

inline bool Slang::Caixa::sobreABorda(Posicao const& c) const {
    int const primeira_linha = origem().linha();
    int const primeira_coluna = origem().coluna();
    int const ultima_linha = destino().linha();
    int const ultima_coluna = destino().coluna();
    return ((c.linha() >= primeira_linha && c.linha() <= ultima_linha && 
	     (c.coluna() == primeira_coluna || c.coluna() == ultima_coluna)) ||
	    (c.coluna() >= primeira_coluna && c.coluna() <= ultima_coluna &&
	     (c.linha() == primeira_linha || c.linha() == ultima_linha)));
}

// Modificadores:
inline void Slang::Caixa::origem(Posicao const & origem) {
    origem_ = origem;
}

inline void Slang::Caixa::destino(Posicao const& destino) {
    *this = Caixa(origem(), destino);
}

inline void Slang::Caixa::dimensao(Dimensao const& dimensao) {
    dimensao_ = dimensao;
}

// Operadores:
inline Slang::Caixa& Slang::Caixa::operator += (Slang::Posicao const& c) {
    origem(origem() + c);
    return *this;
}

inline Slang::Caixa& Slang::Caixa::operator += (Slang::Caixa const& c) {
    if(vazia()) return *this = c;
    if(c.vazia()) return *this;
    int const linha = min(origem_.linha(), c.origem_.linha());
    int const coluna = min(origem_.coluna(), c.origem_.coluna());
    int const linha_final = max(origem_.linha() + dimensao_.linhas(), 
				c.origem_.linha() + c.dimensao_.linhas());
    int const coluna_final = max(origem_.coluna() + dimensao_.colunas(), 
				 c.origem_.coluna() + c.dimensao_.colunas());
    origem_.linha(linha);
    origem_.coluna(coluna);
    dimensao_.linhas(linha_final - linha);
    dimensao_.colunas(coluna_final - coluna);

    return *this;
}

inline Slang::Caixa& Slang::Caixa::operator *= (Slang::Caixa const& c) {
    if(vazia() || c.vazia())
	return *this = caixa_vazia;
    int const linha = max(origem_.linha(), c.origem_.linha());
    int const coluna = max(origem_.coluna(), c.origem_.coluna());
    int const linha_final = min(origem_.linha() + dimensao_.linhas(), 
				c.origem_.linha() + c.dimensao_.linhas());
    int const coluna_final = min(origem_.coluna() + dimensao_.colunas(), 
				 c.origem_.coluna() + c.dimensao_.colunas());
    if(linha_final < linha || coluna_final < coluna)
	return *this = caixa_vazia;
    
    origem_.linha(linha);
    origem_.coluna(coluna);
    dimensao_.linhas(linha_final - linha);
    dimensao_.colunas(coluna_final - coluna);

    return *this;
}
	
// Operadores associados:
inline Slang::Caixa Slang::operator + (Slang::Caixa caixa, 
				       Slang::Posicao const& c) {
    return caixa += c;
}

inline Slang::Caixa Slang::operator + (Slang::Posicao const& c,
				       Slang::Caixa caixa) {
    return caixa += c;
}

inline Slang::Caixa Slang::operator + (Slang::Caixa a, Slang::Caixa const& b) {
    return a += b;
}

inline Slang::Caixa Slang::operator * (Slang::Caixa a, Slang::Caixa const& b) {
    return a *= b;
}


// Mtodos inline da classe ecr:

inline Slang::Ecra::~Ecra() {
    // Terminao do slang:
    SLsmg_reset_smg();
}

inline void Slang::Ecra::atributos(Cor texto, Cor fundo, int objecto) {
    SLtt_set_color(objecto, 0, nome_de_cor[texto], nome_de_cor[fundo]);
}

inline void Slang::Ecra::atributos(Cor texto, Cor fundo) {
    SLtt_set_color(0, 0, nome_de_cor[texto], nome_de_cor[fundo]);
}

inline Slang::Dimensao Slang::Ecra::dimensao() const {
    verificaTamanho();
    return Dimensao(SLtt_Screen_Rows, SLtt_Screen_Cols);
}

inline Slang::Posicao Slang::Ecra::cursor() const {
    return Posicao(SLsmg_get_row(), SLsmg_get_column());
}

inline bool Slang::Ecra::redimensionado() const {
    verificaTamanho();
    if(redimensionado_) {
	redimensionado_ = false;
	return true;
    }
    return false;
}

inline Slang::Ecra::Troco Slang::Ecra::copia(Dimensao const& dimensao) const {
    return copia(Caixa(cursor(), dimensao));
}

inline Slang::Ecra::Troco Slang::Ecra::copia() const {
    return copia(Caixa(Posicao(0, 0), dimensao()));
}

inline void Slang::Ecra::cursor(Posicao const& posicao) {
    verificaTamanho();
    int linha = posicao.linha();
    int coluna = posicao.coluna();
    if(limita_cursor) {
	if(linha >= dimensao().linhas()) linha = dimensao().linhas() - 1;
	else if(linha < 0) linha = 0;
	
	if(coluna >= dimensao().colunas()) coluna = dimensao().colunas() - 1;
	else if(coluna < 0) coluna = 0;
    }
    SLsmg_gotorc(linha, coluna);
}

inline void Slang::Ecra::cursor(Posicao const& posicao) const {
    verificaTamanho();
    int linha = posicao.linha();
    int coluna = posicao.coluna();
    if(limita_cursor) {
	if(linha >= dimensao().linhas()) linha = dimensao().linhas() - 1;
	else if(linha < 0) linha = 0;
	
	if(coluna >= dimensao().colunas()) coluna = dimensao().colunas() - 1;
	else if(coluna < 0) coluna = 0;
    }
    SLsmg_gotorc(linha, coluna);
}

inline void Slang::Ecra::cima() {
    cursor(Posicao(cursor().linha() - 1, cursor().coluna()));
}

inline void Slang::Ecra::baixo() {
    cursor(Posicao(cursor().linha() + 1, cursor().coluna()));
}

inline void Slang::Ecra::esquerda() {
    cursor(Posicao(cursor().linha(), cursor().coluna() - 1));
}

inline void Slang::Ecra::direita() {
    cursor(Posicao(cursor().linha(), cursor().coluna() + 1));
}

inline void Slang::Ecra::desloca(Slang::Tecla const& tecla) {
    switch(tecla) {
      case Tecla::cima:
	cima();
	break;
      case Tecla::baixo:
	baixo();
	break;
      case Tecla::esquerda:
	esquerda();
	break;
      case Tecla::direita:
	direita();
	break;
      default:
	break;
    }
}

inline void Slang::Ecra::campainha() const {
    verificaTamanho();
    SLtt_beep();
}

inline void Slang::Ecra::apagaFimDaLinha() {
    verificaTamanho();
    SLsmg_erase_eol();
}

inline void Slang::Ecra::apagaFimDoEcra() {
    verificaTamanho();
    SLsmg_erase_eos();
}

inline void Slang::Ecra::apaga() {
    verificaTamanho();
    SLsmg_cls();
}

inline void Slang::Ecra::refrescaTudo() const {
    verificaTamanho();
    // Com isto o prximo refresh redesenhar todo o ecr!
    SLsmg_touch_lines(0, dimensao().linhas());
    refresca();
}

inline void Slang::Ecra::refresca() const {
    verificaTamanho();
    SLsmg_refresh();
}

inline void Slang::Ecra::desenhaCaixa(Caixa const& caixa)
{
    verificaTamanho();
    // Guardar cursor inicial:
    Posicao posicao_original_do_cursor;
    if(para_no_proximo)
	posicao_original_do_cursor = cursor();

    SLsmg_draw_box(caixa.origem().linha(), caixa.origem().coluna(), 
		   caixa.dimensao().linhas(), caixa.dimensao().colunas());

    // Regressar ao cursor inicial:
    if(para_no_proximo) {
	cursor(posicao_original_do_cursor);
	para_no_proximo = false;
    }
}

inline Slang::Ecra& Slang::Ecra::operator << (char c) {
    string s;
    s += c;
    return *this << s;
}

inline Slang::Ecra& Slang::Ecra::operator << (int i) {
    // Afazer: usar stringstream (ainda no h em Linux).
    char mal_feito[30];
    std::sprintf(mal_feito, "%d", i);
    return *this << mal_feito;
}

inline Slang::Ecra& Slang::Ecra::operator << (Caixa const& caixa) {
    desenhaCaixa(caixa);
    return *this;
}

inline Slang::Ecra& Slang::Ecra::operator << (Troco const& troco) {
    cola(troco, cursor());
    return *this;
}

inline Slang::Ecra& Slang::Ecra::operator << (Posicao const& posicao) {
    verificaTamanho();
    cursor(posicao);
    return *this;
}

inline Slang::Ecra& Slang::Ecra::operator << (ObjectoCor const& o) {
    verificaTamanho();
    SLsmg_set_color(o.objecto);
    return *this;
}

inline Slang::Ecra& Slang::Ecra::operator << (Fundo const&) {
    verificaTamanho();
    SLsmg_set_color(0);
    return *this;
}

inline Slang::Ecra& Slang::Ecra::operator << (Parado const&) {
    verificaTamanho();
    para_no_proximo = true;
    return *this;
}

inline Slang::Ecra& Slang::Ecra::operator << (Largura const& l) {
    verificaTamanho();
    largura_do_proximo = l.largura;
    return *this;
}

inline Slang::Ecra& Slang::Ecra::operator << (Refresca const&) {
    refresca();
    return *this;
}

inline Slang::Ecra& Slang::Ecra::operator << (RefrescaTudo const&) {
    refrescaTudo();
    return *this;
}

inline Slang::Ecra& Slang::Ecra::operator << (Apaga const&) {
    apaga();
    return *this;
}

inline Slang::Ecra& Slang::Ecra::operator << (ApagaFimDaLinha const&) {
    apagaFimDaLinha();
    return *this;
}

inline Slang::Ecra& Slang::Ecra::operator << (ApagaFimDoEcra const&) {
    apagaFimDoEcra();
    return *this;
}

inline Slang::Ecra& Slang::Ecra::operator << (Campainha const&) {
    campainha();
    return *this;
}

inline void Slang::Ecra::libertaObjecto(int objecto) {
    assert(ocupacao_objecto[objecto] != 0);
    --ocupacao_objecto[objecto];
}

// Mtodos inline da classe Ecra::ObjectoCor:

inline Slang::Ecra::ObjectoCor::ObjectoCor(Cor texto, Cor fundo)
    : texto_(texto), fundo_(fundo) {
    objecto = ecra.reservaObjecto();
    ecra.atributos(texto, fundo, objecto);
}

inline Slang::Ecra::ObjectoCor::~ObjectoCor() {
    ecra.libertaObjecto(objecto);
}

inline Slang::Cor Slang::Ecra::ObjectoCor::texto() const {
    return texto_;
}

inline Slang::Cor Slang:: Ecra::ObjectoCor::fundo() const {
    return fundo_;
}

inline void Slang::Ecra::ObjectoCor::texto(Cor texto) {
    texto_ = texto;
    ecra.atributos(texto_, fundo_, objecto);
}

inline void Slang::Ecra::ObjectoCor::fundo(Cor fundo) {
    fundo_ = fundo;
    ecra.atributos(texto_, fundo_, objecto);
}


// Mtodos inline da classe Ecra::Largura (manipulador):

inline Slang::Ecra::Largura::Largura(int largura) 
    : largura(largura) {
}

inline Slang::Ecra::Largura 
Slang::Ecra::Largura::operator() (int largura) const {
    return Largura(largura);
}


// Mtodos inline da classe Ecra::Troco:

inline Slang::Ecra::Troco::Troco(Dimensao const& dimensao)
    : dimensao_(dimensao), dados(dimensao.linhas() * dimensao.colunas()) {
}

inline Slang::Dimensao Slang::Ecra::Troco::dimensao() const {
    return dimensao_;
}


// Funes e procedimentos no membro inline:

inline Slang::Posicao Slang::cursor(Slang::Posicao const& posicao) {
    return posicao;
}

inline Slang::Posicao Slang::cursor(int linha, int coluna) {
    return Slang::Posicao(linha, coluna);
}

inline Slang::Caixa Slang::caixa(Slang::Posicao const& origem, 
				 Slang::Dimensao const& dimensao) {
    return Caixa(origem, dimensao);
}

inline Slang::Caixa Slang::caixa(int linha, int coluna, 
				 int linhas, int colunas) {
    return Caixa(Posicao(linha, coluna), Dimensao(linhas, colunas));
}
