#ifndef SLANG_MENU_H
#define SLANG_MENU_H

#include <vector>
#include <string>

#include <Slang/ecra.H>

namespace Slang {

    /** Esta classe abstracta serve para definir as ferramentas relacionadas
        com cores para os menus que as possuirem.  Definida no mdulo #menu#
        (ficheiro de interface #Slang/menu.H#).
        
        So definidas cores para a borda, o ttulo, os itens e o item corrente 
        do menu.  As cores usadas so (por omisso):
        \begin{itemize}
        \item borda: preto sobre branco (ou cinzento).
        \item ttulo: vermelho sobre branco (ou cinzento).
        \item itens: preto sobre branco (ou cinzento).
        \item item corrente: amarelo sobre azul.
        \end{itemize}

        Um apndice  uma classe que deve ser vista como abstracta, apesar de
        ser claramente uma classe concreta.  Serva para equipar outras classes 
        de ferramentas mais ou menos avulsas.  Estas classes so conhecidas
        tambm por {\em mixin}.  */
    class ApendiceComCor {
    public:
        /// Construtor da classe (atribui cores por omisso):
        ApendiceComCor();
        
        /** Inspectores */
        //@{
        /// Devolve a cor da borda.
        Ecra::ObjectoCor const& o_borda() const;
        /// Devolve a cor do ttulo.
        Ecra::ObjectoCor const& o_titulo() const;
        /// Devolve a cor dos itens.
        Ecra::ObjectoCor const& o_itens() const;
        /// Devolve a cor do item corrente.
        Ecra::ObjectoCor const& o_corrente() const;
        //@}

        /** Modificadores */
        //@{
        /// Devolve referncia para a cor da borda.
        Ecra::ObjectoCor& o_borda();
        /// Devolve referncia para a cor do ttulo.
        Ecra::ObjectoCor& o_titulo();
        /// Devolve referncia para a cor dos itens.
        Ecra::ObjectoCor& o_itens();
        /// Devolve referncia para a cor do item seleccionado.
        Ecra::ObjectoCor& o_corrente();
        //@}

    private:
        Ecra::ObjectoCor o_borda_;
        Ecra::ObjectoCor o_titulo_;
        Ecra::ObjectoCor o_itens_;
        Ecra::ObjectoCor o_corrente_;
    };

    /** Esta classe abstracta serve para definir a interface bsica de todos
        os menus.  Definida no mdulo #menu# (ficheiro de interface
        #Slang/menu.H#). */
    class Menu {
    public:
        /** Construtor da classe.  
            @param titulo Ttulo do menu. */
        explicit Menu(std::string const& titulo);
        /// Destrutor virtual para poder sofrer derivaes...
        virtual ~Menu();
        /** Funo que executa o menu, devolvendo o nmero da opo escolhida
            pelo utilizador (a primeira opo ou item do menu tem nmero 0). */
        virtual int executa() = 0;
        /// Inspector do ttulo do menu.
        string titulo() const;

    private:
        string titulo_;
    };

    /** Esta classe abstracta serve para definir a interface bsica de todos
        os menus com cores.  A parte das cores  herdada do apndice ({\em
        mixin}) #ApendiceComCor#.  Definida no mdulo #menu# (ficheiro de
        interface #Slang/menu.H#). */
    class MenuComCor: public Menu, public ApendiceComCor {
    public:
        /** Construtor da classe.  
            @param titulo Ttulo do menu. */
        explicit MenuComCor(std::string const& titulo);
    };

    /** Esta classe serve para representar menus simples, que consistem numa
        sequncia de itens.  Definida no mdulo #menu# (ficheiro de
        interface #Slang/menu.H#). 

        {\bf Exemplo de utilizao}

        Este programa mostra um menu simples no ecr e escreve a opo que for
        pressionada at ser seleccionada a opo "Bazar".

        \begin{verbatim}
        #include <Slang/slang.H> 
                
        using namespace Slang; 
        
        int main () 
        { 
            string opcoes[] = {"Bazar",
                               "Nao faz nada...",
                               "Esta tambm no!",
                               "Nem esta..."};
            int numero_de_opcoes = sizeof(opcoes) / sizeof(string);
        
            MenuSimples menu("Um men que no faz nada!", opcoes, 
                             numero_de_opcoes); 
        
            while(int opcao = menu.executa()) 
                ecra << parado << largura(20) << opcoes[opcao] << refresca; 
        }
        \end{verbatim}
    */
    class MenuSimples : public MenuComCor {
    public:
        /** Construtor da classe.
            @param titulo Ttulo do menu.
            @param itens Matriz de cadeias de caracteres com os nomes do
            itens.
            @param numero_de_itens Nmero de itens (e de elementos na matriz).
        */
        MenuSimples(std::string const& titulo, 
                    std::string const itens[], 
                    int numero_de_itens);
        /** Construtor da classe.
            @param titulo Ttulo do menu.
            @param itens Cadeia de caracteres contendo os itens separados por
            fim-de-linha (#'\n'#).
        */
        MenuSimples(std::string const& titulo, 
                    std::string const& itens);
        /** Funo que executa o menu, devolvendo o nmero da opo escolhida
            pelo utilizador (a primeira opo ou item do menu tem nmero 0). */
        virtual int executa();

    private:
        void calculaDimensao();
        void calculaPosicao() const;
        void desenhaItem(int i) const;
        void desenha() const;

        std::vector<std::string> itens;
        int numero_de_itens;
        std::string::size_type largura_maxima;
        mutable Posicao posicao;
        Dimensao dimensao;
        int corrente;
    };

    /** Esta classe serve para representar menus de seleco das cores bsicas 
        usveis no ecr.  Definida no mdulo #menu# (ficheiro de
        interface #Slang/menu.H#). */
    class MenuCor : public MenuSimples {
    public:
        /** Construtor da classe.  
            @param titulo Ttulo do menu. */
        explicit MenuCor(std::string const& titulo);
    };

    /** Esta classe serve para representar menus com apenas duas opes: sim e
        no.  Definida no mdulo #menu# (ficheiro de interface
        #Slang/menu.H#). */
    class MenuSimNao : public MenuSimples {
    public:
        /** Construtor da classe.  
            @param titulo Ttulo do menu. */
        explicit MenuSimNao(std::string const& titulo);
        /** Funo que executa o menu, devolvendo a opo escolhida pelo
            utilizador (1 para sim, 0 para no). */
        virtual int executa();
    };

    /** Esta classe serve para representar caixas de aviso, que apenas mostram
        uma mensagem e das quais se sai pressionando 'enter'.  Definida no
        mdulo #menu# (ficheiro de interface #Slang/menu.H#). */
    class Aviso {
    public:
        /** Construtor da classe.  
            @param mensagem Mensagem a mostrar. */
        explicit Aviso(std::string const& mensagem);
        /// Destrutor virtual para poder sofrer derivaes...
        virtual ~Aviso();
        /// Funo que executa o aviso.
        virtual void executa(); 
    private:
        MenuSimples menu;
    };

    /** Esta classe serve para representar caixas de texto, que quando
        executadas permitem ao utilizador escrever uma cadeia de caracteres.
        Definida no mdulo #menu# (ficheiro de interface #Slang/menu.H#). */
    class CaixaDeTexto : public ApendiceComCor {
    public:
        /** Construtor da classe.  
            @param titulo Ttulo da caixa.
            @param valor Valor inicial da caixa (por omisso a cadeia de
            caracteres vazia).
            @param espacos Nmero de espaos a usar para a edio (pode ser
            espandido se o ttulo for mais largo). */
        explicit CaixaDeTexto(std::string const& titulo, 
                              std::string const& valor = "",
                              std::string::size_type espacos = 0);
        /// Destrutor virtual para poder sofrer derivaes...
        virtual ~CaixaDeTexto();
        /** Funo que executa a caixa de texto, devolvendo a cadeia
            preenchida pelo utilizador. */
        virtual std::string executa(bool impede_vazia = false);
        /// Inspector do ttulo do menu.
        std::string titulo() const;
    private:
        void desenha() const;
        void calculaPosicao() const;
        std::string titulo_;
        string::size_type espacos; // espaos visveis.
        std::string valor;
        string::size_type inicio; // incio da parte visvel da string.
        string::size_type posicao; // posio do cursor nos espaos.
        Dimensao dimensao;
        mutable Posicao origem;
    };
}

#include <Slang/menu_impl.H>

#endif // SLANG_MENU_H
