#ifndef SLANG_TECLADO_H
#define SLANG_TECLADO_H

namespace Slang {

    /** Esta classe serve para representar teclas premidas.  As teclas podem
        corresponder a caracteres (e.g., 'a', 'x', '1', '.') ou a teclas de
        controlo (e.g., 'delete', 'home', etc.). Definida no mdulo #teclado#
        (ficheiro de interface #Slang/teclado.H#). */
    class Tecla {
    public:

        /** Este tipo enumerado representa as vrias teclas que podem ser
            premidas.  No so enumerados os caracteres normais, mas os seus
            valores podem ser representados neste tipo enumerado.  O par de
            tipos #TeclaEnum# e #Tecla# permite usar uma classe como se de um
            tipo enumerado se tratasse.  H converses implcitas entre os
            dois tipos.  Alm disso a classe proporciona algumas funes
            membro teis. */
        enum TeclaEnum {
            /// Tecla de refrescamento ('ctrl-l').
            refresca = 12,
            /// Seta para cima.
            cima = 0x101,
            /// Seta para baixo.
            baixo = 0x102,
            /// Seta para a esquerda.
            esquerda = 0x103,
            /// Seta para a direita.
            direita = 0x104,
            /// Tecla 'page up' ou 'pgina anterior'.
            pagina_anterior = 0x105,
            /// Tecla 'page down' ou 'pgina seguinte'.
            pagina_seguinte = 0x106,
            /// Tecla 'home' ou 'casa'.
            casa = 0x107,
            /// Tecla 'end' ou 'fim'.
            fim = 0x108,
            a1 = 0x109,
            a3 = 0x10A,
            b2 = 0x10B,
            c1 = 0x10C,
            c3 = 0x10D,
            /// Tecla 'redo' ou 'refaz' (disponvel s em alguns teclados).
            refaz = 0x10E,
            /// Tecla 'undo' ou 'desfaz' (disponvel s em alguns teclados).
            desfaz = 0x10F,
            /// Tecla 'backspace' ou 'apaga para trs'.
            apaga_para_tras = 0x110,
            /// Tecla 'enter' ou 'entrada'.
            entrada = 0xD,
            // entrada = 0x111, enter?  No funcionou!
            /// Tecla 'insert' ou 'insere'.
            insere = 0x112,
            /// Tecla 'delete' ou 'apaga'.
            apaga = 0x113,
            F0 = 0x200,
            /// Tecla 'F1'.
            F1 = 0x201,
            /// Tecla 'F2'.
            F2 = 0x202,
            /// Tecla 'F3'.
            F3 = 0x203,
            /// Tecla 'F4'.
            F4 = 0x204,
            /// Tecla 'F5'.
            F5 = 0x205,
            /// Tecla 'F6'.
            F6 = 0x206,
            /// Tecla 'F7'.
            F7 = 0x207,
            /// Tecla 'F8'.
            F8 = 0x208,
            /// Tecla 'F9'.
            F9 = 0x209,
            /// Tecla 'F10'.
            F10 = 0x20A,
            /// Tecla 'F11'.
            F11 = 0x20B,
            /// Tecla 'F12'.
            F12 = 0x20C,
            /// Caractere nulo.
            nulo = 0,
            /// Valor em caso de erro.
            erro = 0xFFFF
        };

        /// Construtor (define converso implcita a partir de #TeclaEnum#).
        Tecla(TeclaEnum tecla);

        /// Operador de converso implcita para #TeclaEnum#.
        operator TeclaEnum () const;

        /** Funo para converso explcita para #char# (s se deve usar se 
            #caractere()# devolver #true#). 
        */
        char comoChar() const;

        /** Devolve #true# se for uma tecla de deslocamento (seta para a
            esquerda, direita, cima ou baixo). */
        bool deslocamento() const;
        /// Devolve #true# se a tecla for um caractere.
        bool caractere() const;

    private:
        TeclaEnum tecla;
    };

    /** Esta classe  um solito: a sua nica instncia representa o
        teclado. Definida no mdulo #teclado# (ficheiro de interface
        #Slang/teclado.H#). 

        A classe #Teclado# permite receber informao sobre a presso de
        qualquer tecla atravs da funo membro #Tecla Teclado::tecla()#, que
        devolve uma instncia da classe #Tecla#. Esta funo no deve ser
        utilizada sem que haja uma tecla disponivel (i.e. sem que o utilizador
        tenha de facto pressionado uma tecla), a no ser que se pretenda
        esperar que o utilizador pressione uma tecla.

        Para verificar se o utilizador carregou numa tecla, existe a funo
        #bool teclaDisponivel(int decimos_de_segundo) const#. Esta funo tem
        como argumento o nmero dcimos de segundo que a funo espera pela
        presso de uma tecla. No deve ser usado zero como argumento desta
        funo, se ela fr usada dentro de um ciclo, dado que pode tornar o
        sistema muito lento (10 dcimos de segundo  um valor aceitvel para o
        argumento desta funo nessas circunstncias).

        A funo membro #void Teclado::limpa()# permite descartar as teclas
        que foram pressionadas pelo utilizador anteriormente e ainda no foram
        lidas e passar a considerar apenas as teclas que forem pressionadas a
        partir desse instante.

         definida uma varivel global chamada #teclado# da classe #Teclado#
        (declarada no ficheiro de interface #Slang/teclado.H#), de modo que
        no  necessrio, nem permitido, criar nenhuma varivel deste tipo nos
        programas que fazem uso desta biblioteca, podendo-se utilizar
        directamente a varivel #teclado# (como acontece no caso do canal
        #cin#).

        Uma instncia da classe #Tecla# pode ser uma tecla especial (definidas
        no enumerado #TeclaEnum#) ou um caracter normal.  possvel verificar
        se o valor armazenado numa instncia da classe #Tecla#  um
        deslocamento ou um caracter normal atravs das funes membro
        #boolTecla::deslocamento()const# e #boolTecla::caractere()const#.

        {\bf Exemplo de utilizao} 

        Este programa captura todas as teclas premidas at ser premido o
        caracter 's'.  Se forem deslocamentos procede ao deslocamento
        respectivo do cursor.  Se forem caracteres escreve-os no ecr.  Se a
        tecla pressionada corresponder a um caractere que no se pode imprimir
        no ecr  mostrada uma mensagem adequada.  Se for outra tecla aparece
        outra mensagem apropriada.
        
        \begin{verbatim}
        #include <Slang/slang.H> 

        using namespace Slang; 

        #include <cctype> // para isprint().

        using namespace std;

        int main () 
        { 
            Ecra::ObjectoCor cor_normal(amarelo, preto); 
            Ecra::ObjectoCor cor_aviso(vermelho, preto); 

            while(true) { 
                if(teclado.teclaDisponivel(10)) { 
                    Tecla a = teclado.tecla(); 

                    if(a.deslocamento())
                        ecra.desloca(a); 
                    else if(a.caractere())
                        if(isprint(a.comoChar()))
                            ecra << cor_normal << a.comoChar();
                        else
                            ecra << cor_aviso << parado 
                                 << "Este nao se pode imprimir!";
                    else
                        ecra << cor_aviso << parado << "Tecla invlida!"; 
                    if(a == 's')
                        break;
                } 
                ecra.refresca(); 
            }
        }
        \end{verbatim}
 */
    class Teclado {
    public:
        Teclado();
        ~Teclado();

        /** Devolve #true# se alguma tecla tiver sido pressionada e portanto
            estiver disponvel para leitura.  Desiste ao fim do tempo passado
            como argumento (em dcimos de segundo). */
        bool teclaDisponivel(int decimos_de_segundo = 0) const;

        /// Devolve a primeira das teclas pressionadas e ainda no lidas.
        Tecla tecla();
        /// Descarta todas as teclas pressionadas e ainda no lidas.
        void limpa();

    private:
        static bool existe_instancia;
    };

    /** Uma varivel global representando o teclado.  A ideia  que substitua
        a varivel global #cin# usada normalmente para ler do teclado. */
    extern Teclado teclado;
}


#include "teclado_impl.H"

#endif // SLANG_TECLADO_H
