#ifndef SLANG_UTIL_H
#define SLANG_UTIL_H

#include <iostream>
#include <string>

namespace Slang {

    class Ignora {
    public:
        Ignora(char terminador, bool limpa = false) 
            : terminador(terminador), limpa(limpa) {
        }
        char terminador;
        bool limpa;
    };
    
    std::istream& operator >> (std::istream& entrada, Ignora const& ignora);

    Ignora const il('\n');
    Ignora const ill('\n', true);

    /**@name Manipuladores extra para #std::istream#
       Definidos no mdulo #menu# (ficheiro de
       interface #Slang/menu.H#).*/
    //@{
    /**@name il
       Manipulador que ignora todos os caracteres at ao prximo fim-de-linha
       (#'\n'#).  O nome  uma abreviatura de "ignora linha".

       {\bf Exemplo de utilizao}

       Se num ficheiro estiverem guardados em linhas consecutivas um inteiro e 
       o nome completo de uma passoa, pode-se tentar ler estes valores como se 
       segue (admite-se que #entrada#  um canal ligado ao ficheiro):
       \begin{verbatim}
       int numero;
       entrada >> numero;
       string nome;
       getline(entrada, nome);
       \end{verbatim}
       Esta soluo no funciona, pois o operador de extraco do inteiro
       deixa o fim-de-linha no canal, o que leva #getline# a ler uma cadeia
       vazia!  A soluo passa por ignorar todos os caracteres at ao
       fim-de-linha:
       \begin{verbatim}
       #include <Slang/util.H>
       using namespace Slang;
       ...
       int numero;
       entrada >> numero >> il;
       string nome;
       getline(entrada, nome);
       \end{verbatim}
       Neste caso o ficheiro de entrada at pode possuir um comentrio depois
       do inteiro, que ser ignorado.  Por exemplo, o ficheiro poderia ser:
       \begin{verbatim}
       12345 Nmero do aluno
       Xisto Ximenes
       \end{verbatim}
       (Que se teria de fazer para que se pudessem colocar comentrios aps o
       nome?) */
    /**@name ill 
       Manipulador que ignora todos os caracteres at ao prximo
       fim-de-linha (#'\n'#) {\bf mas primeiro limpa possveis condies de
       erro do canal}.  O nome  uma abreviatura de "ignora linha limpa".

       {\bf Exemplo de utilizao}

       Suponha que se pretende ler do teclado um inteiro que tem de ser
       no-negativo (o nmero de um aluno, por exemplo).  A soluo bvia :
       \begin{verbatim}
       cout << "Introduza um nmero no negativo: ";
       int numero;
       cin >> numero;
       cout << "O nmero : " << numero << endl;
       \end{verbatim}
       Esta soluo tem dois problemas:
       \begin{itemize}
       \item Se a extraco tiver sucesso, no garante que o valor lido 
       no-negativo.
       \item Se a extraco no tiver sucesso, o canal #cin# fica com uma
       condio de erro, o que faz com que todas as extraces subsequentes
       falhem.
       \end{itemize}
        importante perceber que, neste caso, se assume que ao teclado est um 
       humano, que reconhece e espera poder corrigir os seus erros!  Assim, a
       soluo passa por escrever um ciclo:
       \begin{verbatim}
       int numero;
       while(true) {
           cout << "Introduza um nmero no negativo: ";
           cin >> numero >> il;
           if(numero >= 0)
               break;
           cout << "Tem de ser no negativo!" << endl;
       }
       cout << "O nmero : " << numero << endl;
       \end{verbatim}
       Note-se na utilizao do manipulador #il#, que  usado para que a
       leitura seja orientada por linha.

       Esta soluo resolve o primeiro problema, mas no o segundo...  A
       soluo passa por verificar tambm se a extrao teve sucesso.  Caso
       no tenha tido sucesso  necessrio limpar a condio de erro e ignorar 
       toda a linha.  Dessa forma o utilizador pode voltar a tentar introduzir 
       um nmero:
       \begin{verbatim}
       int numero;
       while(true) {
           cout << "Introduza um nmero no negativo: ";
           if(cin >> numero >> il && numero >= 0)
               break;
           if(cin)
               cout << "Tem de ser no negativo!" << endl;
           else {
               cout << "Isso no  um nmero!" << endl;
               // Ignora resto da linha limpando condio de erro.
               cin >> ill;
           }
       }
       cout << "O nmero : " << numero << endl;
       \end{verbatim}

       Idealmente esta soluo seria encapsulada numa funo que devolvesse o
       nmero e fosse parametrizada pela condio a verificar pelo valor
       (neste caso tem de ser no negativo) e pelas mensagens a escrever no
       ecr. 

       @see il */
    //@}


    /** Esta classe serve de base a uma pequena hierarquia de classes
        representando excepes.  Definida no mdulo #menu# (ficheiro de
        interface #Slang/menu.H#). */
    class Erro {
    public:
        /** Construtor da classe.  
            @param mensagem Uma mensagem explicando a origem da excepo. */
        Erro(std::string const& mensagem)
            : mensagem(mensagem) {
        }
        /// Destrutor virtual para poder sofrer derivaes...   
        virtual ~Erro() {
        }
        /** Inspector da mensagem explicando a origem da excepo na forma de
            uma converso implcita para #std::string#. */
        virtual operator std::string () const {
            return mensagem;
        }
        /// A mensagem explicando a origem da excepo.
        std::string mensagem;
    };

    /** Esta classe serve para representar excepes de carregamento de
        objectos a partir de canais.  Definida no mdulo #menu# (ficheiro de
        interface #Slang/menu.H#). */
    class ErroAoCarregar : public Erro {
    public:
        /** Construtor da classe.  
            @param classe O nome da classe que originou a excepo. */
        ErroAoCarregar(std::string const& classe)
            : Erro(std::string("Erro ao carregar '") + classe + "'") {
        }
    };

    /** Esta classe serve para representar excepes ao guardar objectos
        usando canais.  Definida no mdulo #menu# (ficheiro de interface
        #Slang/menu.H#). */
    class ErroAoGuardar : public Erro {
    public:
        /** Construtor da classe.  
            @param classe O nome da classe que originou a excepo. */
        ErroAoGuardar(std::string const& classe)
            : Erro(std::string("Erro ao guardar '") + classe + "'") {
        }
    };
}

#endif // SLANG_UTIL_H
