#ifndef LISTA_DE_ALUNO_H
#define LISTA_DE_ALUNO_H

#include <iostream>

#include "aluno.H"

/** @brief Representa listas de itens do tipo Item.

    Item  actualmente  um sinnimo de Aluno, mas que pode ser
    alterado facilmente para o tipo que se entender.  Por conveno,
    chama-se "frente" e "tras" ao primeiro e ltimo item na lista.  Os
    nome "primeiro", "ltimo", "incio" e "fim" so reservados para
    iteradores.

    @invariant 0 <= numero_de_itens <= numero_maximo_de_itens. */
class ListaDeAluno {
  public:

    /** @brief Sinnimo de Aluno.  Usa-se para simplificar a tarefa de
	criar listas com itens de outros tipos. */
    typedef Aluno Item;

    /* Declarao de uma classe embutida que serve para percorrer e
       manipular listas: */
    class Iterador;


    // Construtores:

    /// Construtor da classe, cria uma lista vazia.
    ListaDeAluno();


    // Inspectores:

    /** @brief Devolve referncia constante para o item na frente da lista.
	@pre estaVazia(). */
    Item const& frente() const;

    /** @brief Devolve referncia constante para o item na traseira da lista.
	@pre estaVazia(). */
    Item const& tras() const;

    /// Devolve o comprimento da lista, ou seja, o seu nmero de itens.
    int comprimento() const;

    /// Indica se a lista est vazia.
    bool estaVazia() const;

    /// Indica se a lista est cheia.
    bool estaCheia() const;


    // Modificadores:

    /** @brief Devolve referncia para o item na frente da lista.

	No modifica directamente a lista, mas permite modificaes
	atravs da referncia devolvida.

	@pre estaVazia(). */
    Item& frente();

    /** @brief Devolve referncia para o item na traseira da lista.
	@pre estaVazia(). */
    Item& tras();

    /** @brief Poe novo item na frente da lista.
	@pre estaCheia(). */
    void poeNaFrente(Item const& novo_item);

    /** @brief Poe novo item na traseira da lista.
	@pre estaCheia(). */
    void poeAtras(Item const& novo_item);

    /** @brief Insere novo item imediatamente antes da posio indicada pelo
	iterador i.

	Faz com que o iterador continue a referenciar o mesmo item que
	antes da insero.

	@pre estaCheia() e iterador  vlido e iterador <> fim(). */
    void insereAntes(Iterador const& iterador, Item const& novo_item);

    /** @brief Tira o item da frente da lista.
	@pre estaVazia(). */
    void tiraDaFrente();

    /** @brief Tira o item da traseira da lista.
	@pre estaVazia(). */
    void tiraDeTras();

    /// Esvazia a lista.
    void esvazia();

    /** @brief Remove o item indicado pelo iterador i.

	O iterador fica a referenciar o item logo aps o item
	removido.

	@pre iterador  vlido e iterador <> inicio() e iterador <> fim(). */
    void remove(Iterador& iterador);

    /** @brief Remove o item indicado pelo iterador i que fica invlido.

	O iterador  invalidado pela operao!

	@pre iterador  vlido e iterador <> inicio() e iterador <> fim().
	@see remove(Iterador&). */
    void remove(Iterador const& iterador);


    /** @brief Transfere todos itens de outra_lista para a lista
	implcita.

	A outra_lista fica vazia!

	@see operator+=(). */
    void transfereDe(ListaDeAluno& outra_lista);

    /** @brief Concatena a outra_lista com a lista implcita.

	A outra_lista no sofre qualquer alterao.

	@see transfere(). */
    ListaDeAluno& operator+=(ListaDeAluno const& outra_lista);


    /* Funes construtoras de iteradores.  Consideram-se
       modificadoras porque a lista pode ser modificada atravs dos
       iteradores. */

    /** @brief Devolve um novo primeiro iterador, i.e., um iterador
	referenciando o item na frente da lista.

	Note-se que se a lista estiver vazia o primeiro iterador 
	igual ao iterador final. */
    Iterador primeiro();

    /** @brief Devolve um novo ultimo iterador, i.e., um iterador
	referenciando o item na traseira da lista.

	Note-se que se a lista estiver vazia o ltimo iterador  igual
	ao iterador inicial. */
    Iterador ultimo();

    /** @brief Devolve um novo iterador inicial, i.e., um iterador
	referenciando o item fictcio imediatamente antes do item na
	frente da lista. */
    Iterador inicio();

    /** @brief Devolve um novo iterador final, i.e., um iterador
	referenciando o item fictcio imediatamente aps o item na
	traseira da lista. */
    Iterador fim();

    /** @brief Procura a primeira ocorrncia de um dado item.

	Devolve iterador referenciando o item de trs se o item procurado no
	existir.

	@see ultimaOcorrenciaDe(). */
    Iterador primeiraOcorrenciaDe(Item const& item);

    /** @brief Procura a ultimo ocorrncia de um dado item.

	Devolve iterador referenciando o item da frente se o item
	procurado no existir.

	@see primeiraOcorrenciaDe(). */
    Iterador ultimaOcorrenciaDe(Item const& item);

  private:

    // O nmero mximo de itens na lista:
    static int const numero_maximo_de_itens = 100;

    /* A lista  representada por uma cadeia duplamente ligada de elos
       guardados numa matriz de elos.  A matriz de elos contm elos que
       esto na cadeia duplamente ligada (que contm itens da lista) e
       outros elos que esto livres e portanto fazem parte de uma
       cadeia simplesmente ligada.  A cadeia duplamente ligada tem
       guardas (correspondentes aos itens fictcios) mas a cadeia
       simplesmente ligada no tem guardas: */
    struct Elo {
	// O item propriamente dito:
	Item item;
	// Ponteiro para o elo anterior na cadeia:
	Elo* anterior;
	// Ponteiro para o elo seguinte na cadeia:
	Elo* seguinte;
    };

    /* Matriz que guarda os elos das cadeias (incluindo as duas guardas da
       cadeia duplamente ligada de itens): */
    Elo elos[numero_maximo_de_itens + 2];

    // Contador do nmero de itens na lista:
    int numero_de_itens;

    // Ponteiros para as guardas da cadeia duplamente ligada de itens:
    Elo* const elo_inicial;
    Elo* const elo_final;

    /* Ponteiro para o primeiro elo livre na matriz (se no existir nenhum
       contm lixo).   onde comea a cadeia simplesmente ligada dos
       elos livres: */
    Elo* primeiro_elo_livre;

    /* Procedimentos auxiliares para inserir e remover um item da
       cadeia duplamente ligada dos itens: */
    void poe(Elo* const elo_anterior, Elo* const elo_seguinte,
	     Item const& novo_item);
    void tira(Elo* const elo_a_tirar);

    /* Procedimento auxiliar para limpar uma lista, seja qual for o
       seu estado.  Usado pelo construtor da lista e pelo mtodo
       esvazia(). */
    void limpa();

    /* Funo que retira o primeiro elo da cadeia dos elos livres e devolve o
       seu ponteiro.   usada pelo procedimento poe() acima para obter
       um elo livre onde colocar o novo item da lista: */
    Elo* reservaElo();

    /* Procedimento que coloca o elo cujo ponteiro  passado como
       argumento na cadeia dos elos livres.   usado pelo procedimento
       tira() acima quando tira um item da lista (ou melhor, um elo da
       cadeia duplamente ligada): */
    void libertaElo(Elo* const elo_a_libertar);

    /* Funo auxiliar que indica se a condio invariante de
       instncia da classe se verifica: */
    bool cumpreInvariante() const;

    // A classe de iterao tem acesso irrestrito s listas:
    friend class Iterador;
};


/** @brief Representa iteradores para itens de listas do tipo
    ListaDeAluno.

    Os iteradores tm uma caracterstica infeliz: podem estar em
    estados invlidos.	Por exemplo, se uma lista for esvaziada, todos
    os iteradores a ela associada ficam invlidos.   possvel
    resolver este problema, mas  custa de um aumento considervel da
    complexidade deste par de classes.

    @invariant &lista_associada.elos[0] <= elo_do_item_referenciado <
	       &lista_associada[numero_maximo_de_itens+2]. */
class ListaDeAluno::Iterador {
  public:

    // Construtores:

    /** @brief Construtor da classe.
	
	Associa o iterador com a lista passada como argumento e pe-no
	a referenciar o item na sua frente. */
    explicit Iterador(ListaDeAluno& lista_a_associar);


    // Inspectores:

    /* Ateno!  Aqui ocorre a nica mudana na interface dos
       iteradores.  Na realidade a alterao no  necessria. Mas
       introduz uma conveniente simplificao notacional.  Alis,
       poder-se-ia ter optado por uma soluo menos drstica: manter o
       inspector item() original e fornecer em paralelo o operador *,
       por uma questo de convenincia. */

    /** @brief Devolve uma referncia para o item referenciado pelo
	iterador.

	Note-se que a referncia devolvida no  constante.   que um
	iterador const no pode ser alterado (avanar ou recuar), mas
	permite alerar o item por ele referenciado na lista
	associada.

	@pre O item referenciado no pode ser nenhum dos itens
	fictcios da lista (i.e., nem o item antes da frente da lista,
	nem o item aps a sua traseira). */
    Item& operator*() const;

    /** @brief Indica se dois iteradores so iguais.

	Ou melhor, se a instncia implcita  igual ao iterador passado como
	argumento.  Dois iteradores so iguais se se referirem ao mesmo item
	da mesma lista (mesmo que sejam itens fictcios). A comparao s faz
	sentido se os iteradores estiverem associados  mesma lista! */
    bool operator==(Iterador const& outro_iterador) const;

    /// Operador de diferena entre iteradores.
    bool operator!=(Iterador const& outro_iterador) const;


    // Modificadores:

    /** @brief Avana iterador para o prximo item da lista.  Devolve o
	prprio iterador.
	@pre O iterador no pode ser o iterador final da lista. */
    Iterador& operator++();

    /** @brief Recua iterador para o item anterior da lista.  Devolve o
	prprio iterador.
	@pre O iterador no pode ser o iterador inicial da lista. */
    Iterador& operator--();

    /** @brief Avana iterador para o prximo item da lista.  Devolve um novo
	iterador com o valor do prprio iterador antes de avanado.
	@pre O iterador no pode ser o iterador final da lista. */
    Iterador operator++(int);

    /** @brief Recua iterador para o item anterior da lista.  Devolve um novo
	iterador com o valor do prprio iterador antes de recuado.
	@pre O iterador no pode ser o iterador inicial da lista. */
    Iterador operator--(int);


  private:

    // Ponteiro para a lista a que o iterador est associado:
    ListaDeAluno* lista_associada;

    // Ponteiro para o elo do item da lista referenciado pelo iterador:
    Elo* elo_do_item_referenciado;

    /* Construtor privado da classe.  Associa o iterador com a
       lista_a_associar e pe-no a referenciar o item no elo
       indicado. */
    Iterador(ListaDeAluno& lista_a_associar,
	     Elo* const elo_do_item_a_referenciar);

    /* Inspector para obter o ponteiro para o elo do item referenciado pelo
       iterador: */
    Elo* elo() const;

    // Inspector para obter a lista associada ao iterador:
    ListaDeAluno const* lista() const;

    // Funo auxiliar que indica se a condio invariante de
    // instncia da classe se verifica:
    bool cumpreInvariante() const;

    /* A classe ListaDeAluno tem acesso irrestrito a todos os membros
       da classe Iterador.   importante perceber que as duas classes,
       ListaDeAluno e ListaDeAluno::Iterador esto completamente
       interligadas.  No h qualquer promiscuidade nesta relao.
       So partes do mesmo todo. */
    friend class ListaDeAluno;
};


/** @brief Operador de insero de listas num canal.

    Todos os itens da lista so inseridos por ordem entre parnteses e
    separados por vrgulas.

    Usa-se ListaDeAluno& em vez de ListaDeAluno const&, porque caso
    constrrio ter-se-ia de definir uma classe de iterador para listas
    constantes (coisa que ser feita mais tarde). */
std::ostream& operator<<(std::ostream& saida, ListaDeAluno& lista);


#include "lista_de_aluno_impl.H"

#endif // LISTA_DE_ALUNO_H
