#include <cassert>

inline ListaDeTelefonema::ListaDeTelefonema()
{
    limpa();

    assert(cumpreInvariante());
}

inline ListaDeTelefonema::Item const& ListaDeTelefonema::frente() const
{
    assert(cumpreInvariante());
    assert(not estaVazia());

    return elos[elos[elo_inicial].seguinte].item;
}

inline ListaDeTelefonema::Item const& ListaDeTelefonema::tras() const
{
    assert(cumpreInvariante());
    assert(not estaVazia());

    return elos[elos[elo_final].anterior].item;
}

inline int ListaDeTelefonema::comprimento() const
{
    assert(cumpreInvariante());

    return numero_de_itens;
}

inline bool ListaDeTelefonema::estaVazia() const
{
    assert(cumpreInvariante());

    return comprimento() == 0;
}

inline bool ListaDeTelefonema::estaCheia() const
{
    assert(cumpreInvariante());

    return comprimento() == numero_maximo_de_itens;
}

inline ListaDeTelefonema::Item& ListaDeTelefonema::frente()
{
    assert(cumpreInvariante());
    assert(not estaVazia());

    return elos[elos[elo_inicial].seguinte].item;
}

inline ListaDeTelefonema::Item& ListaDeTelefonema::tras()
{
    assert(cumpreInvariante());
    assert(not estaVazia());

    return elos[elos[elo_final].anterior].item;
}

/* Os mtodos poeNaFrente(), poeAtras() e insere() so todos
   implementados  custa do mtodo membro privado poe(): */

inline void ListaDeTelefonema::poeNaFrente(Item const& novo_item)
{
    assert(cumpreInvariante());
    assert(not estaCheia());

    poe(elo_inicial, elos[elo_inicial].seguinte, novo_item);

    assert(cumpreInvariante());
}

inline void ListaDeTelefonema::poeAtras(Item const& novo_item)
{
    assert(cumpreInvariante());
    assert(not estaCheia());

    poe(elos[elo_final].anterior, elo_final, novo_item);

    assert(cumpreInvariante());
}

inline void ListaDeTelefonema::insereAntes(Iterador const& iterador,
					   Item const& novo_item)
{
    assert(cumpreInvariante());
    assert(not estaCheia());
    /* No se pode inserir se o iterador referir o item antes do da
       frente (item fictcio inicial): */
    assert(iterador != inicio());

    /* S se pode inserir se o iterador estiver associado a esta
       lista!  Mas para o verificar  necessria matria da prxima
       aula... */
    // assert(iterador associado  instncia implcita...);

    poe(elos[iterador.elo_do_item_referenciado].anterior,
	iterador.elo_do_item_referenciado, novo_item);

    assert(cumpreInvariante());
}

/* Os mtodos tiraDaFrente(), tiraDeTras() e remove() so todos
   implementados  custa do mtodo privado tira(): */

inline void ListaDeTelefonema::tiraDaFrente()
{
    assert(cumpreInvariante());
    assert(not estaVazia());

    tira(elos[elo_inicial].seguinte);

    assert(cumpreInvariante());
}

inline void ListaDeTelefonema::tiraDeTras()
{
    assert(cumpreInvariante());
    assert(not estaVazia());

    tira(elos[elo_final].anterior);

    assert(cumpreInvariante());
}

inline void ListaDeTelefonema::remove(Iterador& iterador)
{
    assert(cumpreInvariante());
    assert(iterador != inicio() and iterador != fim());

    int elo_a_remover = iterador.elo_do_item_referenciado;

    ++iterador;

    tira(elo_a_remover);

    assert(cumpreInvariante());
}

inline void ListaDeTelefonema::esvazia()
{
    assert(cumpreInvariante());

    limpa();

    assert(cumpreInvariante());
}

inline ListaDeTelefonema::Iterador ListaDeTelefonema::primeiro()
{
    assert(cumpreInvariante());

    // Cria-se um iterador para esta lista, que referencia
    // inicialmente o item na frente da lista (ver construtor de
    // ListaDeTelefonema::Iterador), e devolve-se imediatamente o
    // iterador criado:
    return Iterador(*this);
}

inline ListaDeTelefonema::Iterador ListaDeTelefonema::ultimo()
{
    assert(cumpreInvariante());

    // Cria-se um iterador para esta lista:
    Iterador iterador(*this);

    // O iterador deve referenciar item na traseira da lista:
    iterador.elo_do_item_referenciado = elos[elo_final].anterior;

    /* Em bom rigor no  boa ideia que seja um mtodo da lista a
       verificar se a codio invariante de instncia do iterador 
       verdadeira...  Mais tarde se ver melhor forma de resolver o
       problema. */
    assert(iterador.cumpreInvariante());
    assert(cumpreInvariante());

    return iterador;
}

inline ListaDeTelefonema::Iterador ListaDeTelefonema::inicio()
{
    assert(cumpreInvariante());

    Iterador iterador(*this);

    // O iterador deve referenciar o item inicial, i.e., o item
    // fictcio antes da frente da lista:
    iterador.elo_do_item_referenciado = elo_inicial;

    assert(iterador.cumpreInvariante());
    assert(cumpreInvariante());

    return iterador;
}

inline ListaDeTelefonema::Iterador ListaDeTelefonema::fim()
{
    assert(cumpreInvariante());

    Iterador iterador(*this);

    // O iterador deve referenciar o item final, i.e., o item fictcio
    // aps a traseira da lista:
    iterador.elo_do_item_referenciado = elo_final;

    assert(iterador.cumpreInvariante());
    assert(cumpreInvariante());

    return iterador;
}

inline int ListaDeTelefonema::reservaElo()
{
    assert(not estaCheia());

    int elo_reservado = primeiro_elo_livre;
    primeiro_elo_livre = elos[primeiro_elo_livre].seguinte;

    return elo_reservado;
}

inline void ListaDeTelefonema::libertaElo(int const elo_a_libertar)
{
    elos[elo_a_libertar].seguinte = primeiro_elo_livre;
    primeiro_elo_livre = elo_a_libertar;
}

/* Coloca um novo elo, contendo o item novo_item e retirado da cadeia
   de elos livres, na cadeia duplamente ligada.	 O novo elo  colocado
   entre o elo dado por anterior e o elo dado por seguinte. */
inline void ListaDeTelefonema::poe(int const elo_anterior,
				   int const elo_seguinte,
				   Item const& novo_item)
{
    assert(elos[elo_anterior].seguinte == elo_seguinte and
	   elos[elo_seguinte].anterior == elo_anterior);

    assert(not estaCheia());

    ++numero_de_itens;

    // Obter novo elo:
    int const elo_reservado = reservaElo();

    // Guardar novo item:
    elos[elo_reservado].item = novo_item;

    // Indices do novo elo passam a referir elos anterior e seguinte
    // dados:
    elos[elo_reservado].anterior = elo_anterior;
    elos[elo_reservado].seguinte = elo_seguinte;

    // ndices dos elos anterior e seguinte passam a referir-se ao
    // novo elo:
    elos[elo_anterior].seguinte = elo_reservado;
    elos[elo_seguinte].anterior = elo_reservado;
}

// Retira o elo_a_tirar da cadeia duplamente ligada, colocando-o na
// cadeia simplesmente ligada de elos livres:
inline void ListaDeTelefonema::tira(int const elo_a_tirar)
{
    assert(not estaVazia());

    --numero_de_itens;

    elos[elos[elo_a_tirar].anterior].seguinte = elos[elo_a_tirar].seguinte;
    elos[elos[elo_a_tirar].seguinte].anterior = elos[elo_a_tirar].anterior;

    libertaElo(elo_a_tirar);
}

inline bool ListaDeTelefonema::cumpreInvariante() const
{
    /* O ideal era verificar aqui se as cadeias esto em bom estado.
       Para isso podia-se marcar os elos livres com um anterior
       valendo -1, por exemplo.	 Depois era s verificar se:

       a) A cadeia dos ocupados  consistente.
       b) Todos os elos pertencem ou  cadeia dos livres ou  dos
       ocupados.

       Para simplificar no se fazem essas verificaes. */

    return 0 <= numero_de_itens and numero_de_itens <= numero_maximo_de_itens;
}


inline
ListaDeTelefonema::Iterador::Iterador(ListaDeTelefonema& lista_a_associar)
    : lista_associada(lista_a_associar),
      elo_do_item_referenciado(
	  lista_a_associar.elos[lista_a_associar.elo_inicial].seguinte)
{
    assert(cumpreInvariante());
}

inline ListaDeTelefonema::Item& ListaDeTelefonema::Iterador::item() const
{
    assert(cumpreInvariante());
    assert(*this != lista_associada.inicio() and
	   *this != lista_associada.fim());

    return lista_associada.elos[elo_do_item_referenciado].item;
}

/* Comparam-se os ndices.  Esta implementao no  perfeita.	Em
   rigor dever-se-ia verificar ambos os iteradores esto associados 
   mesma lista, pois no faz sentido compar-los de outra forma (pense
   como faz-lo quando tiver aprendido ponteiros). */
inline bool
ListaDeTelefonema::Iterador::operator==(Iterador const& outro_iterador) const
{
    assert(cumpreInvariante() and outro_iterador.cumpreInvariante());
    // assert(iteradores associados  mesma lista...);

    return elo_do_item_referenciado == outro_iterador.elo_do_item_referenciado;
}

inline bool
ListaDeTelefonema::Iterador::operator!=(Iterador const& outro_iterador) const
{
    assert(cumpreInvariante() and outro_iterador.cumpreInvariante());
    // assert(iteradores associados  mesma lista...);

    return not (*this == outro_iterador);
}

inline ListaDeTelefonema::Iterador& ListaDeTelefonema::Iterador::operator++()
{
    assert(cumpreInvariante());
    assert(*this != lista_associada.fim());

    elo_do_item_referenciado =
	lista_associada.elos[elo_do_item_referenciado].seguinte;

    assert(cumpreInvariante());

    return *this;
}

inline ListaDeTelefonema::Iterador& ListaDeTelefonema::Iterador::operator--()
{
    assert(cumpreInvariante());
    assert(*this != lista_associada.inicio());

    elo_do_item_referenciado =
	lista_associada.elos[elo_do_item_referenciado].anterior;

    assert(cumpreInvariante());

    return *this;
}

inline ListaDeTelefonema::Iterador ListaDeTelefonema::Iterador::operator++(int)
{
    assert(cumpreInvariante());
    assert(*this != lista_associada.fim());

    ListaDeTelefonema::Iterador resultado = *this;
    ++*this;

    assert(cumpreInvariante());

    return resultado;
}

inline ListaDeTelefonema::Iterador ListaDeTelefonema::Iterador::operator--(int)
{
    assert(cumpreInvariante());
    assert(*this != lista_associada.inicio());

    ListaDeTelefonema::Iterador resultado = *this;
    --*this;

    assert(cumpreInvariante());

    return resultado;
}

inline bool ListaDeTelefonema::Iterador::cumpreInvariante() const
{
    /* Em rigor dever-se-ia verificar se o indice do elo pertence a um
       elo ocupado, pois de outra forma o iterador est corrompido.
       Usando o critrio proposto atrs de marcar os elos livres com
       um anterior especial, esta verificao seria trivial. */
    return 0 <= elo_do_item_referenciado and
	elo_do_item_referenciado < numero_maximo_de_itens + 2;
}
