#include <cassert>

inline ListaDeAluno::ListaDeAluno()
    : elo_inicial(elos + numero_maximo_de_itens),
      elo_final(elos + numero_maximo_de_itens + 1)
{
    limpa();

    assert(cumpreInvariante());
}

inline ListaDeAluno::Item const& ListaDeAluno::frente() const
{
    assert(cumpreInvariante());
    assert(not estaVazia());

    return elo_inicial->seguinte->item;
}

inline ListaDeAluno::Item const& ListaDeAluno::tras() const
{
    assert(cumpreInvariante());
    assert(not estaVazia());

    return elo_final->anterior->item;
}

inline int ListaDeAluno::comprimento() const
{
    assert(cumpreInvariante());

    return numero_de_itens;
}

inline bool ListaDeAluno::estaVazia() const
{
    assert(cumpreInvariante());

    return comprimento() == 0;
}

inline bool ListaDeAluno::estaCheia() const
{
    assert(cumpreInvariante());

    return comprimento() == numero_maximo_de_itens;
}

inline ListaDeAluno::Item& ListaDeAluno::frente()
{
    assert(cumpreInvariante());
    assert(not estaVazia());

    return elo_inicial->seguinte->item;
}

inline ListaDeAluno::Item& ListaDeAluno::tras()
{
    assert(cumpreInvariante());
    assert(not estaVazia());

    return elo_final->anterior->item;
}

/* Os mtodos poeNaFrente(), poeAtras() e insere() so todos
   implementados  custa do mtodo membro privado poe(): */

inline void ListaDeAluno::poeNaFrente(Item const& novo_item)
{
    assert(cumpreInvariante());
    assert(not estaCheia());

    poe(elo_inicial, elo_inicial->seguinte, novo_item);

    assert(cumpreInvariante());
}

inline void ListaDeAluno::poeAtras(Item const& novo_item)
{
    assert(cumpreInvariante());
    assert(not estaCheia());

    poe(elo_final->anterior, elo_final, novo_item);

    assert(cumpreInvariante());
}

inline void ListaDeAluno::insereAntes(Iterador const& iterador,
				      Item const& novo_item)
{
    assert(cumpreInvariante());
    assert(not estaCheia());
    assert(this == iterador.lista());
    assert(iterador != inicio());

    poe(iterador.elo()->anterior, iterador.elo(), novo_item);

    assert(cumpreInvariante());
}

/* Os mtodos tiraDaFrente(), tiraDeTras() e remove() so todos
   implementados  custa do mtodo privado tira(): */

inline void ListaDeAluno::tiraDaFrente()
{
    assert(cumpreInvariante());
    assert(not estaVazia());

    tira(elo_inicial->seguinte);

    assert(cumpreInvariante());
}

inline void ListaDeAluno::tiraDeTras()
{
    assert(cumpreInvariante());
    assert(not estaVazia());

    tira(elo_final->anterior);

    assert(cumpreInvariante());
}

inline void ListaDeAluno::remove(Iterador& iterador)
{
    assert(cumpreInvariante());
    assert(iterador != inicio() and iterador != fim());
    assert(this == iterador.lista());

    Elo* const elo_a_remover = iterador.elo();

    ++iterador;

    tira(elo_a_remover);

    assert(cumpreInvariante());
}

inline void ListaDeAluno::remove(Iterador const& iterador)
{
    assert(cumpreInvariante());
    assert(iterador != inicio() and iterador != fim());
    assert(this == iterador.lista());

    tira(iterador.elo());

    assert(cumpreInvariante());
}

inline void ListaDeAluno::esvazia()
{
    assert(cumpreInvariante());

    /* Esta  uma soluo simplista, pois o limpa reconstitui a cadeia
       simplesmente ligada de elos livres, o que no  necessrio.  Para
       listas quase vazias este mtodo  ineficiente. */
    limpa();

    assert(cumpreInvariante());
}

inline ListaDeAluno::Iterador ListaDeAluno::primeiro()
{
    assert(cumpreInvariante());

    // Cria-se um iterador para esta lista, que referencia
    // inicialmente o item na frente da lista (ver construtor de
    // ListaDeAluno::Iterador), e devolve-se imediatamente o
    // iterador criado:
    return Iterador(*this);
}

inline ListaDeAluno::Iterador ListaDeAluno::ultimo()
{
    assert(cumpreInvariante());

    return Iterador(*this, elo_final->anterior);
}

inline ListaDeAluno::Iterador ListaDeAluno::inicio()
{
    assert(cumpreInvariante());

    return Iterador(*this, elo_inicial);
}

inline ListaDeAluno::Iterador ListaDeAluno::fim()
{
    assert(cumpreInvariante());

    return Iterador(*this, elo_final);
}

inline ListaDeAluno::Elo* ListaDeAluno::reservaElo()
{
    assert(not estaCheia());

    Elo* const elo_reservado = primeiro_elo_livre;
    primeiro_elo_livre = elo_reservado->seguinte;

    return elo_reservado;
}

inline void ListaDeAluno::libertaElo(Elo* const elo_a_libertar)
{
    elo_a_libertar->seguinte = primeiro_elo_livre;
    primeiro_elo_livre = elo_a_libertar;
}

/* Coloca um novo elo, contendo o item novo_item e retirado da cadeia
   de elos livres, na cadeia duplamente ligada.	 O novo elo  colocado
   entre o elo dado por anterior e o elo dado por seguinte. */
inline void ListaDeAluno::poe(Elo* const elo_anterior,
				   Elo* const elo_seguinte,
				   Item const& novo_item)
{
    assert(elo_anterior->seguinte == elo_seguinte and
	   elo_seguinte->anterior == elo_anterior);
    assert(not estaCheia());

    ++numero_de_itens;

    // Obter novo elo:
    Elo* const elo_reservado = reservaElo();

    elo_reservado->item = novo_item;

    elo_reservado->anterior = elo_anterior;
    elo_reservado->seguinte = elo_seguinte;

    elo_anterior->seguinte = elo_reservado;
    elo_seguinte->anterior = elo_reservado;
}

// Retira o elo_a_tirar da cadeia duplamente ligada, colocando-o na
// cadeia simplesmente ligada de elos livres:
inline void ListaDeAluno::tira(Elo* const elo_a_tirar)
{
    assert(not estaVazia());

    --numero_de_itens;

    elo_a_tirar->anterior->seguinte = elo_a_tirar->seguinte;
    elo_a_tirar->seguinte->anterior = elo_a_tirar->anterior;

    libertaElo(elo_a_tirar);
}

inline bool ListaDeAluno::cumpreInvariante() const
{
    /* O ideal era verificar aqui se as cadeias esto em bom estado.
       Para isso podia-se marcar os elos livres com um anterior
       valendo -1, por exemplo.	 Depois era s verificar se:
       a) A cadeia dos ocupados  consistente.
       b) Todos os elos pertencem ou  cadeia dos livres ou  dos
       ocupados.
       Para simplificar no se fazem essas verificaes. */

    return 0 <= numero_de_itens and numero_de_itens <= numero_maximo_de_itens;
}


inline
ListaDeAluno::Iterador::Iterador(ListaDeAluno& lista_a_associar)
    : lista_associada(&lista_a_associar),
      elo_do_item_referenciado(lista_a_associar.elo_inicial->seguinte)
{
    assert(cumpreInvariante());
}

inline ListaDeAluno::Item& ListaDeAluno::Iterador::operator*() const
{
    assert(cumpreInvariante());

    // O item s existe se o iterador no se referir a um item fictcio:
    assert(*this != lista_associada->inicio() and
	   *this != lista_associada->fim());

    return elo_do_item_referenciado->item;
}

inline bool
ListaDeAluno::Iterador::operator==(Iterador const& outro_iterador) const
{
    assert(cumpreInvariante() and outro_iterador.cumpreInvariante());
    assert(lista_associada == outro_iterador.lista_associada);

    return elo_do_item_referenciado == outro_iterador.elo_do_item_referenciado;
}

inline bool
ListaDeAluno::Iterador::operator!=(Iterador const& outro_iterador) const
{
    assert(cumpreInvariante() and outro_iterador.cumpreInvariante());
    assert(lista_associada == outro_iterador.lista_associada);

    return not(*this == outro_iterador);
}

inline ListaDeAluno::Iterador& ListaDeAluno::Iterador::operator++()
{
    assert(cumpreInvariante());
    assert(*this != lista_associada->fim());

    elo_do_item_referenciado = elo_do_item_referenciado->seguinte;

    assert(cumpreInvariante());
    return *this;
}

inline ListaDeAluno::Iterador& ListaDeAluno::Iterador::operator--()
{
    assert(cumpreInvariante());
    assert(*this != lista_associada->inicio());

    elo_do_item_referenciado = elo_do_item_referenciado->anterior;

    assert(cumpreInvariante());
    return *this;
}

inline ListaDeAluno::Iterador ListaDeAluno::Iterador::operator++(int)
{
    assert(cumpreInvariante());
    assert(*this != lista_associada->fim());

    ListaDeAluno::Iterador resultado = *this;
    ++*this;

    assert(cumpreInvariante());

    return resultado;
}

inline ListaDeAluno::Iterador ListaDeAluno::Iterador::operator--(int)
{
    assert(cumpreInvariante());
    assert(*this != lista_associada->inicio());

    ListaDeAluno::Iterador resultado = *this;
    --*this;

    assert(cumpreInvariante());

    return resultado;
}

inline bool ListaDeAluno::Iterador::cumpreInvariante() const
{
    /* Em rigor dever-se-ia verificar se o elo pertence  cadeia dos
       ocupados, pois de outra forma o iterador est corrompido.
       Usando o critrio proposto atrs de marcar os elos livres com
       um anterior especial, esta verificao seria trivial. */
    return (lista_associada->elos <= elo_do_item_referenciado and
	    elo_do_item_referenciado < (lista_associada->elos +
					numero_maximo_de_itens + 2));
}

inline
ListaDeAluno::Iterador::Iterador(ListaDeAluno& lista_a_associar,
				      Elo* const elo_do_item_a_referenciar)
    : lista_associada(&lista_a_associar),
      elo_do_item_referenciado(elo_do_item_a_referenciar)
{
    assert(cumpreInvariante());
}

inline ListaDeAluno const* ListaDeAluno::Iterador::lista() const
{
    assert(cumpreInvariante());

    return lista_associada;
}

inline ListaDeAluno::Elo* ListaDeAluno::Iterador::elo() const
{
    assert(cumpreInvariante());

    return elo_do_item_referenciado;
}
