#include "lista_de_int.H"

ListaDeInt::ListaDeInt(ListaDeInt const& original)
    : numero_de_itens(0), elo_inicial(new Elo)
{
    // Captura-se qualquer excepo, pois reimplementaes podem levar o
    // construtor dos itens a lanar outro tipo de excepo que no bad_alloc.
    try {
	elo_inicial->seguinte = elo_final = new Elo(elo_inicial);
    } catch(...) {
	delete elo_inicial;
	throw;
    }

    // A primeira parte em nada difere do construtor por omisso.

    // A segunda faz a cpia dos itens:

    // Captura-se qualquer excepo, pois reimplementaes podem levar
    // poeAtras() a lanar outro tipo de excepo que no bad_alloc, e
    // alm disso pode ser o construtor de algum item a falhar...
    try {
	for(Elo* elo = original.elo_inicial->seguinte;
	    elo != original.elo_final;
	    elo = elo->seguinte)
	    poeAtras(elo->item);
    } catch(...) {
	// Se no se conseguiu colocar algum item, a construo falhou
	// e  necessrio destruir tudo o que entretanto foi
	// construdo:
	esvazia();
	delete elo_inicial;
	delete elo_final;

	// Finalmente  necessrio relanar a excepo!	 Afinal, houve
	// um erro e por isso a construo falhou...
	throw;
    }

    assert(cumpreInvariante());
}

/* Aqui podia-se resolver o problema de vrias formas.	Com
   iteradores, por exemplo.  Ou chamando tiraDaFrente() at a lista
   estar vazia.	 A soluo escolhida  da baixo nvel, e  tambm mais
   eficiente...	 Mas  a que exige maiores alteraes se se mudar a
   implementao da lista... */
void ListaDeInt::esvazia()
{
    assert(cumpreInvariante());

    numero_de_itens = 0;

    // Destruir elos dinmicos:
    for(Elo* elo = elo_inicial->seguinte; elo != elo_final; ) {
	Elo* seguinte = elo->seguinte;
	delete elo;
	elo = seguinte;
    }

    // A cadeia duplamente ligada tem de ficar vazia:
    elo_inicial->seguinte = elo_final;
    elo_final->anterior = elo_inicial;

    assert(cumpreInvariante());
}

ListaDeInt::IteradorConstante ListaDeInt::
primeiraOcorrenciaDe(Item const& item) const
{
    IteradorConstante i = primeiro();
    while(i != fim() && *i != item)
	++i;

    return i;
}

ListaDeInt::IteradorConstante
ListaDeInt::ultimaOcorrenciaDe(Item const& item) const
{
    IteradorConstante i = ultimo();
    while(i != inicio() && *i != item)
	--i;

    return i;
}

ListaDeInt::Iterador ListaDeInt::primeiraOcorrenciaDe(Item const& item)
{
    Iterador i = primeiro();
    while(i != fim() and *i != item)
	++i;

    return i;
}

ListaDeInt::Iterador ListaDeInt::ultimaOcorrenciaDe(Item const& item)
{
    Iterador i = ultimo();
    while(i != inicio() and *i != item)
	--i;

    return i;
}

std::ostream& operator<<(std::ostream& saida, ListaDeInt const& lista)
{
    saida << '(';
    for(ListaDeInt::IteradorConstante i = lista.primeiro();
	i != lista.fim(); ++i) {
	saida << *i;
	if(i != lista.ultimo())
	    saida << ",";
    }
    return saida << ')';
}


#ifdef TESTE

#include <iostream>
#include <fstream>

using namespace std;

/** Teste de unidade do TAD ListaDeInts.  Verifica o bom
    funcionamento de todas as operaes sem excepo! */
int main()
{
    ListaDeInt lista;

    assert(lista.comprimento() == 0);

    lista.poeNaFrente(212);
    lista.poeAtras(3123);
    lista.poeNaFrente(11);
    lista.poeAtras(41234);

    assert(lista.comprimento() == 4);

    assert(lista.frente() == 11);
    assert(lista.tras() == 41234);

    // No se pode reaproveitar os iteradores, pois no so
    // atribuveis entre si.  Por isso cada bloco de teste  colocado
    // entre chavetas:

    {
	ListaDeInt::Iterador i = lista.primeiro();
	assert(i == lista.primeiro());
	assert(*i == 11);
	++i;
	assert(*i == 212);
	ListaDeInt::Iterador j1 = ++i;
	assert(j1 == i);
	assert(*i == 3123);
	ListaDeInt::Iterador j2 = i++;
	assert(j2 == j1);
	assert(i == lista.ultimo());
	assert(*i == 41234);
	++i;
	assert(i == lista.fim());
    }

    assert(lista.ultimo() != lista.inicio());
    lista.esvazia();
    assert(lista.estaVazia());
    assert(lista.comprimento() == 0);
    assert(lista.primeiro() == lista.fim());
    assert(lista.ultimo() == lista.inicio());

    lista.poeNaFrente(212);
    lista.poeAtras(3123);
    lista.poeNaFrente(11);
    lista.poeAtras(41234);

    {
	ListaDeInt::Iterador i = lista.ultimo();
	assert(i == lista.ultimo());
	++i;
	assert(i == lista.fim());
	--i;
	assert(*i == 41234);
	ListaDeInt::Iterador j1 = --i;
	assert(j1 == i);
	assert(*i == 3123);
	ListaDeInt::Iterador j2 = i--;
	assert(j2 == j1);
	assert(*i == 212);
	--i;
	assert(i == lista.primeiro());
	assert(*i == 11);
	--i;
	assert(i == lista.inicio());
	++i;
	assert(i == lista.primeiro());
    }

    {
	ListaDeInt::Iterador i = lista.primeiro();
	++i;
	lista.insereAntes(i, 1010);
	assert(lista.comprimento() == 5);
	assert(*i == 212);
	--i;
	--i;
	assert(i == lista.primeiro());
	assert(*i == 11);
	++i;
	assert(*i == 1010);
	++i;
	assert(*i == 212);
	++i;
	assert(*i == 3123);
	++i;
	assert(i == lista.ultimo());
	assert(*i == 41234);
	++i;
	assert(i == lista.fim());
    }

    {
	ListaDeInt::Iterador i = lista.primeiro();
	++i;
	++i;
	lista.remove(i);
	assert(lista.comprimento() == 4);
	assert(*i == 3123);
    }

    {
	ListaDeInt::Iterador i = lista.primeiro();
	assert(i == lista.primeiro());
	assert(*i == 11);
	++i;
	assert(*i == 1010);
	++i;
	assert(*i == 3123);
	++i;
	assert(i == lista.ultimo());
	assert(*i == 41234);
	++i;
	assert(i == lista.fim());
    }

    lista.tiraDaFrente();
    lista.tiraDeTras();
    assert(lista.comprimento() == 2);
    assert(lista.frente() == 1010);
    assert(lista.tras() == 3123);

    {
	ListaDeInt::Iterador i = lista.primeiro();
	assert(*i == 1010);
	++i;
	assert(*i == 3123);
	++i;
	assert(i == lista.fim());
    }

    assert(not lista.estaVazia());

    lista.tiraDaFrente();
    lista.tiraDeTras();
    assert(lista.comprimento() == 0);
    assert(lista.estaVazia());

    lista.poeNaFrente(212);
    lista.poeAtras(3123);
    lista.poeNaFrente(11);
    lista.poeAtras(41234);

    lista.insereAntes(lista.primeiraOcorrenciaDe(3123),
		      21121);

    assert(lista.comprimento() == 5);

    {
	ListaDeInt::Iterador i = lista.ultimo();
	-- --i;
	assert(*i == 21121);

	++ ++i;

	i = lista.primeiraOcorrenciaDe(21121);
	lista.insereAntes(i, 20120);

	assert(lista.comprimento() == 6);

	assert(*i == 21121);

	--i;

	assert(*i == 20120);
    }

    ListaDeInt nova_lista = lista;

    {
	ListaDeInt::Iterador i = lista.primeiro();
	ListaDeInt::Iterador j = nova_lista.primeiro();
	
	while(i != lista.fim() and j != nova_lista.fim()) {
	    assert(*i == *j);
	    ++i;
	    ++j;
	}

	assert(i == lista.fim() and j == nova_lista.fim());
    }

    nova_lista.tiraDaFrente();

    assert(lista.frente() == 11);

    lista = nova_lista;

    {
	ListaDeInt::Iterador i = lista.primeiro();
	ListaDeInt::Iterador j = nova_lista.primeiro();
	
	while(i != lista.fim() and j != nova_lista.fim()) {
	    assert(*i == *j);
	    ++i;
	    ++j;
	}

	assert(i == lista.fim() and j == nova_lista.fim());
    }

    ListaDeInt outra_lista = nova_lista;

    outra_lista += nova_lista;

    {
	ListaDeInt::Iterador i = nova_lista.primeiro();
	ListaDeInt::Iterador j = outra_lista.primeiro();
	
	while(i != nova_lista.fim() and j != outra_lista.fim()) {
	    assert(*i == *j);
	    ++i;
	    ++j;
	}

	assert(i == nova_lista.fim() and j != outra_lista.fim());

	i = nova_lista.primeiro();
	while(i != nova_lista.fim() and j != outra_lista.fim()) {
	    assert(*i == *j);
	    ++i;
	    ++j;
	}

	assert(i == nova_lista.fim() and j == outra_lista.fim());
    }

    ListaDeInt ainda_outra_lista;
    ainda_outra_lista.transfereDe(outra_lista);

    assert(outra_lista.estaVazia());

    {
	ListaDeInt::Iterador i = lista.primeiro();
	ListaDeInt::Iterador j = ainda_outra_lista.primeiro();
	
	while(i != lista.fim() and j != ainda_outra_lista.fim()) {
	    assert(*i == *j);
	    ++i;
	    ++j;
	}

	assert(i == lista.fim() and j != ainda_outra_lista.fim());

	i = lista.primeiro();

	while(i != lista.fim() and j != ainda_outra_lista.fim()) {
	    assert(*i == *j);
	    ++i;
	    ++j;
	}

	assert(i == lista.fim() and j == ainda_outra_lista.fim());
    }

    lista.esvazia();
    lista.poeAtras(11);
    lista += lista;
    lista.transfereDe(lista);

    assert(lista.comprimento() == 2);
    assert(lista.frente() == 11);
    assert(lista.tras() == 11);

    ListaDeInt lista_recomecada;

    lista_recomecada.poeNaFrente(212);
    lista_recomecada.poeAtras(3123);
    lista_recomecada.poeNaFrente(11);
    lista_recomecada.poeAtras(41234);

    ListaDeInt const lista_constante = lista_recomecada;

    assert(not lista_constante.estaVazia());
    assert(lista_constante.frente() == 11);
    assert(lista_constante.tras() == 41234);

    {
	ListaDeInt::IteradorConstante i = lista_constante.primeiro();
	ListaDeInt::Iterador j = lista_recomecada.primeiro();
	
	while(i != lista_constante.fim() and j != lista_recomecada.fim()) {
	    assert(*i == *j);
	    ++i;
	    ++j;
	}
	
	assert(i == lista_constante.fim() and j == lista_recomecada.fim());
    }

    {
	ListaDeInt::IteradorConstante i = lista_constante.primeiro();
	ListaDeInt::Iterador j = lista_recomecada.primeiro();
	
	while(i != lista_constante.fim() and j != lista_recomecada.fim()) {
	    assert(*i == *j);
	    i++;
	    ++j;
	}
	
	assert(i == lista_constante.fim() and j == lista_recomecada.fim());
    }

    ListaDeInt::Iterador i = lista_recomecada.primeiro();
    ListaDeInt::IteradorConstante j = lista_recomecada.primeiro();

    assert(i == j);

    lista_recomecada.insereAntes(j, 512345);
    --i;
    assert(*i == 512345);
    assert(*j == 11);

    lista_recomecada.remove(j);
    ++i;
    assert(*i == 212);
    assert(*j == 212);

    ListaDeInt::IteradorConstante const ic = i;
    ListaDeInt::Iterador const jc = ++i;
    lista_recomecada.tiraDaFrente();
    assert(lista_recomecada.comprimento() == 3);

    lista_recomecada.remove(ic);
    lista_recomecada.remove(jc);

    assert(lista_recomecada.comprimento() == 1);
    assert(lista_recomecada.frente() == 41234);
    assert(lista_recomecada.tras() == 41234);
}

#endif // TESTE
