#include <cassert>

inline ListaDeTelefonema::ListaDeTelefonema()
    : elo_inicial(elos + numero_maximo_de_itens),
      elo_final(elos + numero_maximo_de_itens + 1)
{
    limpa();

    assert(cumpreInvariante());
}

inline ListaDeTelefonema::Item const& ListaDeTelefonema::frente() const
{
    assert(cumpreInvariante());
    assert(not estaVazia());

    return elo_inicial->seguinte->item;
}

inline ListaDeTelefonema::Item const& ListaDeTelefonema::tras() const
{
    assert(cumpreInvariante());
    assert(not estaVazia());

    return elo_final->anterior->item;
}

inline int ListaDeTelefonema::comprimento() const
{
    assert(cumpreInvariante());

    return numero_de_itens;
}

inline bool ListaDeTelefonema::estaVazia() const
{
    assert(cumpreInvariante());

    return comprimento() == 0;
}

inline bool ListaDeTelefonema::estaCheia() const
{
    assert(cumpreInvariante());

    return comprimento() == numero_maximo_de_itens;
}

inline ListaDeTelefonema::Item& ListaDeTelefonema::frente()
{
    assert(cumpreInvariante());
    assert(not estaVazia());

    return elo_inicial->seguinte->item;
}

inline ListaDeTelefonema::Item& ListaDeTelefonema::tras()
{
    assert(cumpreInvariante());
    assert(not estaVazia());

    return elo_final->anterior->item;
}

/* Os mtodos poeNaFrente(), poeAtras() e insere() so todos
   implementados  custa do mtodo membro privado poe(): */

inline void ListaDeTelefonema::poeNaFrente(Item const& novo_item)
{
    assert(cumpreInvariante());
    assert(not estaCheia());

    poe(elo_inicial, elo_inicial->seguinte, novo_item);

    assert(cumpreInvariante());
}

inline void ListaDeTelefonema::poeAtras(Item const& novo_item)
{
    assert(cumpreInvariante());
    assert(not estaCheia());

    poe(elo_final->anterior, elo_final, novo_item);

    assert(cumpreInvariante());
}

inline void ListaDeTelefonema::insereAntes(Iterador const& iterador,
					   Item const& novo_item)
{
    assert(cumpreInvariante());
    assert(not estaCheia());
    // S se pode inserir se o iterador estiver associado a esta lista!
    assert(this == iterador.lista());
    /* No se pode inserir se o iterador referir o item antes do da frente
       (item fictcio inicial): */
    assert(iterador != inicio());

    poe(iterador.elo()->anterior, iterador.elo(), novo_item);

    assert(cumpreInvariante());
}

/* Os mtodos tiraDaFrente(), tiraDeTras() e remove() so todos
   implementados  custa do mtodo privado tira(): */

inline void ListaDeTelefonema::tiraDaFrente()
{
    assert(cumpreInvariante());
    assert(not estaVazia());

    tira(elo_inicial->seguinte);

    assert(cumpreInvariante());
}

inline void ListaDeTelefonema::tiraDeTras()
{
    assert(cumpreInvariante());
    assert(not estaVazia());

    tira(elo_final->anterior);

    assert(cumpreInvariante());
}

inline void ListaDeTelefonema::remove(Iterador& iterador)
{
    assert(cumpreInvariante());
    assert(iterador != inicio() and iterador != fim());
    // S se pode remover se o iterador estiver associado a esta lista!
    assert(this == iterador.lista());

    Elo* const elo_a_remover = iterador.elo();

    ++iterador;

    tira(elo_a_remover);

    assert(cumpreInvariante());
}

inline void ListaDeTelefonema::remove(Iterador const& iterador)
{
    assert(cumpreInvariante());
    assert(iterador != inicio() and iterador != fim());
    assert(this == iterador.lista());

    tira(iterador.elo());

    assert(cumpreInvariante());
}

inline void ListaDeTelefonema::esvazia()
{
    assert(cumpreInvariante());

    /* Esta  uma soluo simplista, pois o limpa reconstitui a cadeia
       simplesmente ligada de elos livres, o que no  necessrio.  Para
       listas quase vazias este mtodo  ineficiente. */
    limpa();

    assert(cumpreInvariante());
}

inline ListaDeTelefonema::Iterador ListaDeTelefonema::primeiro()
{
    assert(cumpreInvariante());

    // Cria-se um iterador para esta lista, que referencia
    // inicialmente o item na frente da lista (ver construtor de
    // ListaDeTelefonema::Iterador), e devolve-se imediatamente o
    // iterador criado:
    return Iterador(*this);
}

inline ListaDeTelefonema::Iterador ListaDeTelefonema::ultimo()
{
    assert(cumpreInvariante());

    return Iterador(*this, elo_final->anterior);
}

inline ListaDeTelefonema::Iterador ListaDeTelefonema::inicio()
{
    assert(cumpreInvariante());

    return Iterador(*this, elo_inicial);
}

inline ListaDeTelefonema::Iterador ListaDeTelefonema::fim()
{
    assert(cumpreInvariante());

    return Iterador(*this, elo_final);
}

inline ListaDeTelefonema::Elo* ListaDeTelefonema::reservaElo()
{
    assert(not estaCheia());

    Elo* const elo_reservado = primeiro_elo_livre;
    primeiro_elo_livre = elo_reservado->seguinte;

    return elo_reservado;
}

inline void ListaDeTelefonema::libertaElo(Elo* const elo_a_libertar)
{
    elo_a_libertar->seguinte = primeiro_elo_livre;
    primeiro_elo_livre = elo_a_libertar;
}

/* Coloca um novo elo, contendo o item novo_item e retirado da cadeia
   de elos livres, na cadeia duplamente ligada.	 O novo elo  colocado
   entre o elo dado por anterior e o elo dado por seguinte. */
inline void ListaDeTelefonema::poe(Elo* const elo_anterior,
				   Elo* const elo_seguinte,
				   Item const& novo_item)
{
    assert(elo_anterior->seguinte == elo_seguinte and
	   elo_seguinte->anterior == elo_anterior);
    assert(not estaCheia());

    ++numero_de_itens;

    // Obter novo elo:
    Elo* const elo_reservado = reservaElo();

    elo_reservado->item = novo_item;

    elo_reservado->anterior = elo_anterior;
    elo_reservado->seguinte = elo_seguinte;

    elo_anterior->seguinte = elo_reservado;
    elo_seguinte->anterior = elo_reservado;
}

// Retira o elo_a_tirar da cadeia duplamente ligada, colocando-o na
// cadeia simplesmente ligada de elos livres:
inline void ListaDeTelefonema::tira(Elo* const elo_a_tirar)
{
    assert(not estaVazia());

    --numero_de_itens;

    elo_a_tirar->anterior->seguinte = elo_a_tirar->seguinte;
    elo_a_tirar->seguinte->anterior = elo_a_tirar->anterior;

    libertaElo(elo_a_tirar);
}

inline bool ListaDeTelefonema::cumpreInvariante() const
{
    /* O ideal era verificar aqui se as cadeias esto em bom estado.
       Para isso podia-se marcar os elos livres com um anterior
       valendo -1, por exemplo.	 Depois era s verificar se:
       a) A cadeia dos ocupados  consistente.
       b) Todos os elos pertencem ou  cadeia dos livres ou  dos
       ocupados.
       Para simplificar no se fazem essas verificaes. */

    return 0 <= numero_de_itens and numero_de_itens <= numero_maximo_de_itens;
}


inline
ListaDeTelefonema::Iterador::Iterador(ListaDeTelefonema& lista_a_associar)
    : lista_associada(&lista_a_associar),
      elo_do_item_referenciado(lista_a_associar.elo_inicial->seguinte)
{
    assert(cumpreInvariante());
}

inline ListaDeTelefonema::Item& ListaDeTelefonema::Iterador::item() const
{
    assert(cumpreInvariante());

    // O item s existe se o iterador no se referir a um item fictcio:
    assert(*this != lista_associada->inicio() and
	   *this != lista_associada->fim());

    return elo_do_item_referenciado->item;
}

inline bool
ListaDeTelefonema::Iterador::operator==(Iterador const& outro_iterador) const
{
    assert(cumpreInvariante() and outro_iterador.cumpreInvariante());
    assert(lista_associada == outro_iterador.lista_associada);

    return elo_do_item_referenciado == outro_iterador.elo_do_item_referenciado;
}

inline bool
ListaDeTelefonema::Iterador::operator!=(Iterador const& outro_iterador) const
{
    assert(cumpreInvariante() and outro_iterador.cumpreInvariante());
    assert(lista_associada == outro_iterador.lista_associada);

    return not(*this == outro_iterador);
}

inline ListaDeTelefonema::Iterador& ListaDeTelefonema::Iterador::operator++()
{
    assert(cumpreInvariante());
    assert(*this != lista_associada->fim());

    elo_do_item_referenciado = elo_do_item_referenciado->seguinte;

    assert(cumpreInvariante());
    return *this;
}

inline ListaDeTelefonema::Iterador& ListaDeTelefonema::Iterador::operator--()
{
    assert(cumpreInvariante());
    assert(*this != lista_associada->inicio());

    elo_do_item_referenciado = elo_do_item_referenciado->anterior;

    assert(cumpreInvariante());
    return *this;
}

inline ListaDeTelefonema::Iterador ListaDeTelefonema::Iterador::operator++(int)
{
    assert(cumpreInvariante());
    assert(*this != lista_associada->fim());

    ListaDeTelefonema::Iterador resultado = *this;
    ++*this;

    assert(cumpreInvariante());

    return resultado;
}

inline ListaDeTelefonema::Iterador ListaDeTelefonema::Iterador::operator--(int)
{
    assert(cumpreInvariante());
    assert(*this != lista_associada->inicio());

    ListaDeTelefonema::Iterador resultado = *this;
    --*this;

    assert(cumpreInvariante());

    return resultado;
}

inline bool ListaDeTelefonema::Iterador::cumpreInvariante() const
{
    /* Em rigor dever-se-ia verificar se o elo pertence  cadeia dos
       ocupados, pois de outra forma o iterador est corrompido.
       Usando o critrio proposto atrs de marcar os elos livres com
       um anterior especial, esta verificao seria trivial. */
    return (lista_associada->elos <= elo_do_item_referenciado and
	    elo_do_item_referenciado < (lista_associada->elos +
					numero_maximo_de_itens + 2));
}

inline
ListaDeTelefonema::Iterador::Iterador(ListaDeTelefonema& lista_a_associar,
				      Elo* const elo_do_item_a_referenciar)
    : lista_associada(&lista_a_associar),
      elo_do_item_referenciado(elo_do_item_a_referenciar)
{
    assert(cumpreInvariante());
}

inline ListaDeTelefonema const* ListaDeTelefonema::Iterador::lista() const
{
    assert(cumpreInvariante());

    return lista_associada;
}

inline ListaDeTelefonema::Elo* ListaDeTelefonema::Iterador::elo() const
{
    assert(cumpreInvariante());

    return elo_do_item_referenciado;
}
