#ifndef LISTA_DE_TELEFONEMA_H
#define LISTA_DE_TELEFONEMA_H

#include <iostream>
#include "telefonema.H"

/** @brief Representa listas de itens do tipo Item.

    Item  actualmente  um sinnimo de Telefonema, mas que pode ser
    alterado facilmente para o tipo que se entender.  Por conveno,
    chama-se "frente" e "tras" ao primeiro e ltimo item na lista.  Os
    nome "primeiro", "ltimo", "incio" e "fim" so reservados para
    iteradores.

    @invariant 0 <= numero_de_itens <= numero_maximo_de_itens. */
class ListaDeTelefonema {
  public:

    /** @brief Sinnimo de Telefonema.	Usa-se para simplificar a tarefa
	de criar listas com itens de outros tipos. */
    typedef Telefonema Item;

    /* Declarao de uma classe embutida que serve para percorrer e
       manipular listas: */
    class Iterador;


    // Construtores:

    /// Construtor da classe, cria uma lista vazia.
    ListaDeTelefonema();


    // Inspectores:

    /** @brief Devolve referncia constante para o item na frente da lista.
	@pre estaVazia(). */
    Item const& frente() const;

    /** @brief Devolve referncia constante para o item na traseira da lista.
	@pre estaVazia(). */
    Item const& tras() const;

    /// Devolve o comprimento da lista, ou seja, o seu nmero de itens.
    int comprimento() const;

    /// Indica se a lista est vazia.
    bool estaVazia() const;

    /// Indica se a lista est cheia.
    bool estaCheia() const;


    // Modificadores:

    /** @brief Devolve referncia para o item na frente da lista.

	No modifica directamente a lista, mas permite modificaes
	atravs da referncia devolvida.

	@pre estaVazia(). */
    Item& frente();

    /** @brief Devolve referncia para o item na traseira da lista.
	@pre estaVazia(). */
    Item& tras();

    /** @brief Poe novo item na frente da lista.
	@pre estaCheia(). */
    void poeNaFrente(Item const& novo_item);

    /** @brief Poe novo item na traseira da lista.
	@pre estaCheia(). */
    void poeAtras(Item const& novo_item);

    /** @brief Insere novo item imediatamente antes da posio indicada pelo
	iterador i.

	Faz com que o iterador continue a referenciar o mesmo item que
	antes da insero.

	@pre estaCheia() e iterador  vlido e iterador <> fim(). */
    void insereAntes(Iterador& iterador, Item const& novo_item);

    /** @brief Tira o item da frente da lista.
	@pre estaVazia(). */
    void tiraDaFrente();

    /** @brief Tira o item da traseira da lista.
	@pre estaVazia(). */
    void tiraDeTras();

    /// Esvazia a lista.
    void esvazia();

    /** @brief Remove o item indicado pelo iterador i.

	O iterador fica a referenciar o item logo aps o item
	removido.

	@pre iterador  vlido e iterador <> inicio() e iterador <> fim(). */
    void remove(Iterador& iterador);


    /* Funes construtoras de iteradores.  Consideram-se
       modificadoras porque a lista pode ser modificada atravs dos
       iteradores. */

    /** @brief Devolve um novo primeiro iterador, i.e., um iterador
	referenciando o item na frente da lista.

	Note-se que se a lista estiver vazia o primeiro iterador 
	igual ao iterador final. */
    Iterador primeiro();

    /** @brief Devolve um novo ultimo iterador, i.e., um iterador
	referenciando o item na traseira da lista.

	Note-se que se a lista estiver vazia o ltimo iterador  igual
	ao iterador inicial. */
    Iterador ultimo();

    /** @brief Devolve um novo iterador inicial, i.e., um iterador
	referenciando o item fictcio imediatamente antes do item na
	frente da lista. */
    Iterador inicio();

    /** @brief Devolve um novo iterador final, i.e., um iterador
	referenciando o item fictcio imediatamente aps o item na
	traseira da lista. */
    Iterador fim();

  private:
    // O nmero mximo de itens na lista:
    static int const numero_maximo_de_itens = 100;

    // Matriz que guarda os itens da lista:
    Item itens[numero_maximo_de_itens];

    // Contador do nmero de itens na lista:
    int numero_de_itens;

    /* Funo auxiliar que indica se a condio invariante de
       instncia da classe se verifica: */
    bool cumpreInvariante() const;

    // A classe de iterao tem acesso irrestrito s listas:
    friend class Iterador;
};


/** @brief Representa iteradores para itens de listas do tipo
    ListaDeTelefonema.

    Os iteradores tm uma caracterstica infeliz: podem estar em
    estados invlidos.	Por exemplo, se uma lista for esvaziada, todos
    os iteradores a ela associada ficam invlidos.   possvel
    resolver este problema, mas  custa de um aumento considervel da
    complexidade deste par de classes.

    @invariant -1 <= indice_do_item_referenciado.  Note-se que no se pode
    garantir que em cada instante indice_do_item_referenciado seja inferior ou
    igual ao nmero de itens na lista, pois a lista pode ser alterada
    independentemente do iterador! */
class ListaDeTelefonema::Iterador {
  public:

    // Construtores:

    /** @brief Construtor da classe.

	Associa o iterador com a lista passada como argumento e pe-no
	a referenciar o item na sua frente. */
    explicit Iterador(ListaDeTelefonema& lista_a_associar);


    // Inspectores:

    /** @brief Devolve uma referncia para o item referenciado pelo
	iterador.

	Note-se que a referncia devolvida no  constante.   que um
	iterador const no pode ser alterado (avanar ou recuar), mas
	permite alerar o item por ele referenciado na lista
	associada.

	@pre O item referenciado no pode ser nenhum dos itens
	fictcios da lista (i.e., nem o item antes da frente da lista,
	nem o item aps a sua traseira). */
    Item& item() const;

    /** @brief Indica se dois iteradores so iguais.

	Ou melhor, se a instncia implcita  igual ao iterador passado
	como argumento.	 Dois iteradores so iguais se se referirem ao
	mesmo item da mesma lista (mesmo que sejam itens fictcios). */
    bool operator==(Iterador const& outro_iterador) const;

    /// Operador de diferena entre iteradores.
    bool operator!=(Iterador const& outro_iterador) const;


    // Modificadores:

    /** @brief Avana iterador para o prximo item da lista.  Devolve o
	prprio iterador.
	@pre O iterador no pode ser o iterador final da lista. */
    Iterador& operator++();

    /** @brief Recua iterador para o item anterior da lista.  Devolve o
	prprio iterador.
	@pre O iterador no pode ser o iterador inicial da lista. */
    Iterador& operator--();

    /** @brief Avana iterador para o prximo item da lista.  Devolve um novo
	iterador com o valor do prprio iterador antes de avanado.
	@pre O iterador no pode ser o iterador final da lista. */
    Iterador operator++(int);

    /** @brief Recua iterador para o item anterior da lista.  Devolve um novo
	iterador com o valor do prprio iterador antes de recuado.
	@pre O iterador no pode ser o iterador inicial da lista. */
    Iterador operator--(int);


  private:
    // Referncia para a lista a que o iterador est associado:
    ListaDeTelefonema& lista_associada;

    // ndice do item da lista referenciado pelo iterador:
    int indice_do_item_referenciado;

    // Funo auxiliar que indica se a condio invariante de
    // instncia da classe se verifica:
    bool cumpreInvariante() const;

    /* A classe ListaDeTelefonema tem acesso irrestrito a todos os membros
       da classe Iterador.   importante perceber que as duas classes,
       ListaDeTelefonema e ListaDeTelefonema::Iterador esto completamente
       interligadas.  No h qualquer promiscuidade nesta relao.
       So partes do mesmo todo. */
    friend class ListaDeTelefonema;
};


/** @brief Operador de insero de listas num canal.

    Todos os itens da lista so inseridos por ordem entre parnteses e
    separados por vrgulas.

    Usa-se ListaDeTelefonema& em vez de ListaDeTelefonema const&, porque caso
    constrrio ter-se-ia de definir uma classe de iterador para listas
    constantes (coisa que ser feita mais tarde). */
std::ostream& operator<<(std::ostream& saida, ListaDeTelefonema& lista);


#include "lista_de_telefonema_impl.H"

#endif // LISTA_DE_TELEFONEMA_H
