#include <cassert>

inline ListaDeTelefonema::ListaDeTelefonema()
    : numero_de_itens(0)
{
    assert(cumpreInvariante());
}

inline ListaDeTelefonema::Item const& ListaDeTelefonema::frente() const
{
    assert(cumpreInvariante());
    assert(not estaVazia());

    return itens[0];
}

inline ListaDeTelefonema::Item const& ListaDeTelefonema::tras() const
{
    assert(cumpreInvariante());
    assert(not estaVazia());

    return itens[numero_de_itens - 1];
}

inline int ListaDeTelefonema::comprimento() const
{
    assert(cumpreInvariante());

    return numero_de_itens;
}

inline bool ListaDeTelefonema::estaVazia() const
{
    assert(cumpreInvariante());

    return comprimento() == 0;
}

inline bool ListaDeTelefonema::estaCheia() const
{
    assert(cumpreInvariante());

    return comprimento() == numero_maximo_de_itens;
}

inline ListaDeTelefonema::Item& ListaDeTelefonema::frente()
{
    assert(cumpreInvariante());
    assert(not estaVazia());

    return itens[0];
}

inline ListaDeTelefonema::Item& ListaDeTelefonema::tras()
{
    assert(cumpreInvariante());
    assert(not estaVazia());

    return itens[numero_de_itens - 1];
}

inline void ListaDeTelefonema::poeAtras(Item const& novo_item)
{
    assert(cumpreInvariante());
    assert(not estaCheia());

    itens[numero_de_itens] = novo_item;
    ++numero_de_itens;

    assert(cumpreInvariante());
}

inline void ListaDeTelefonema::tiraDeTras()
{
    assert(cumpreInvariante());
    assert(not estaVazia());

    --numero_de_itens;

    assert(cumpreInvariante());
}

inline void ListaDeTelefonema::esvazia()
{
    assert(cumpreInvariante());

    numero_de_itens = 0;

    assert(cumpreInvariante());
}

inline ListaDeTelefonema::Iterador ListaDeTelefonema::primeiro()
{
    assert(cumpreInvariante());

    // Cria-se um iterador para esta lista, que referencia
    // inicialmente o item na frente da lista (ver construtor de
    // ListaDeTelefonema::Iterador), e devolve-se imediatamente o
    // iterador criado:
    return Iterador(*this);
}

inline ListaDeTelefonema::Iterador ListaDeTelefonema::ultimo()
{
    assert(cumpreInvariante());

    // Cria-se um iterador para esta lista:
    Iterador iterador(*this);

    // O iterador deve referenciar item na traseira da lista:
    iterador.indice_do_item_referenciado = comprimento() - 1;

    /* Em bom rigor no  boa ideia que seja um mtodo da lista a
       verificar se a codio invariante de instncia do iterador 
       verdadeira...  Mais tarde se ver melhor forma de resolver o
       problema. */
    assert(iterador.cumpreInvariante());
    assert(cumpreInvariante());

    return iterador;
}

inline ListaDeTelefonema::Iterador ListaDeTelefonema::inicio()
{
    assert(cumpreInvariante());

    Iterador iterador(*this);

    // O iterador deve referenciar o item inicial, i.e., o item
    // fictcio antes da frente da lista:
    iterador.indice_do_item_referenciado = -1;

    assert(iterador.cumpreInvariante());
    assert(cumpreInvariante());

    return iterador;
}

inline ListaDeTelefonema::Iterador ListaDeTelefonema::fim()
{
    assert(cumpreInvariante());

    Iterador iterador(*this);

    // O iterador deve referenciar o item final, i.e., o item fictcio
    // aps a traseira da lista:
    iterador.indice_do_item_referenciado = comprimento();

    assert(iterador.cumpreInvariante());
    assert(cumpreInvariante());

    return iterador;
}

inline bool ListaDeTelefonema::cumpreInvariante() const
{
    return 0 <= numero_de_itens and numero_de_itens <= numero_maximo_de_itens;
}


inline
ListaDeTelefonema::Iterador::Iterador(ListaDeTelefonema& lista_a_associar)
    : lista_associada(lista_a_associar), indice_do_item_referenciado(0)
{

    assert(cumpreInvariante());
}

inline ListaDeTelefonema::Item& ListaDeTelefonema::Iterador::item() const
{
    assert(cumpreInvariante());

    // O item s existe se o iterador no se referir a um item fictcio:
    assert(*this != lista_associada.inicio() and
	   *this != lista_associada.fim());

    return lista_associada.itens[indice_do_item_referenciado];
}

/* Comparam-se os ndices.  Esta implementao no  perfeita.	Em
   rigor dever-se-ia verificar ambos os iteradores esto associados 
   mesma lista, pois no faz sentido compar-los de outra forma (pense
   como faz-lo quando tiver aprendido ponteiros). */
inline bool
ListaDeTelefonema::Iterador::operator==(Iterador const& outro_iterador) const
{
    assert(cumpreInvariante() and outro_iterador.cumpreInvariante());
    // assert(iteradores associados  mesma lista...);

    return indice_do_item_referenciado ==
	outro_iterador.indice_do_item_referenciado;
}

inline bool
ListaDeTelefonema::Iterador::operator!=(Iterador const& outro_iterador) const
{
    assert(cumpreInvariante() and outro_iterador.cumpreInvariante());
    // assert(iteradores associados  mesma lista...);

    return not (*this == outro_iterador);
}

inline ListaDeTelefonema::Iterador& ListaDeTelefonema::Iterador::operator++()
{
    assert(cumpreInvariante());

    // Incrementar para alm do item fictcio final  um erro:
    assert(*this != lista_associada.fim());

    ++indice_do_item_referenciado;

    assert(cumpreInvariante());

    return *this;
}

inline ListaDeTelefonema::Iterador& ListaDeTelefonema::Iterador::operator--()
{
    assert(cumpreInvariante());
    assert(*this != lista_associada.inicio());

    --indice_do_item_referenciado;

    assert(cumpreInvariante());

    return *this;
}

inline ListaDeTelefonema::Iterador ListaDeTelefonema::Iterador::operator++(int)
{
    assert(cumpreInvariante());
    assert(*this != lista_associada.fim());

    ListaDeTelefonema::Iterador resultado = *this;
    ++*this;

    assert(cumpreInvariante());

    return resultado;
}

inline ListaDeTelefonema::Iterador ListaDeTelefonema::Iterador::operator--(int)
{
    assert(cumpreInvariante());
    assert(*this != lista_associada.inicio());

    ListaDeTelefonema::Iterador resultado = *this;
    --*this;

    assert(cumpreInvariante());

    return resultado;
}

inline bool ListaDeTelefonema::Iterador::cumpreInvariante() const
{
    return -1 <= indice_do_item_referenciado;
}
