#ifndef TELEFONEMA_H
#define TELEFONEMA_H

#include <string>
#include <iostream>

/** Representa um telefonema.
    @invariant 0 <= duracao_. */
class Telefonema {
  public:
    /** Constri um novo telefonema para um dado nmero e com uma dada durao.
	 construtor por omisso no porque faa sentido em geral, mas porque a
	isso obriga, por enquanto, a sua utilizao nas listas.
	@pre 0 <= duracao.
	@post numero() = numero e duracao() = duracao. */
    Telefonema(std::string const& numero = "", int duracao = 0);

    /** Constri um novo telefonema extraindo a respectiva informao de um
	canal de entrada.  Note-se que esta operao deveria lanar uma 
	excepo no caso de a leitura no ter sucesso.  A ver mais tarde.
	@pre canal_de_entrada.good().
	@post numero() e duracao() so o nmero e a durao que estavam 
	      presentes no canal de entrada (que deixou de os conter). */
    Telefonema(std::istream& canal_de_entrada);

    /** Devolve o nmero do telefonema.
	@pre V.
	@post numero = nmero do telefonema. */
    std::string const& numero() const;

    /** Devolve a durao do telefonema.
	@pre V.
	@post duracao = durao do telefonema. */
    int duracao() const;

    /** Guarda informao sobre o telefonema num canal de sada.
	@pre canal_de_saida.good().
	@post canal_de_saida.fail() ou 
	      (canal_de_saida.fail() e canal_de_saida contm o nmero e a 
	      durao). */
    void guardaEm(std::ostream& canal_de_saida) const;

  private:
    std::string numero_;
    int duracao_;    

    /** Indica se a condio invariante da classe  verdadeira.
	@pre V.
	@post 0 <= duracao_. */
    bool cumpreInvariante() const;
};

/** Indica se dois telefonemas so iguais, i.e., se dizem respeito ao mesmo 
    nmero e tm a mesma durao.
    @pre V.
    @post operator== = (um_telefonema.numero() == outro_telefonema.numero() e
                        um_telefonema.duracao() == 
			outro_telefonema.duracao()). */
bool operator==(Telefonema const& um_telefonema,
		Telefonema const& outro_telefonema);

/** Indica se dois telefonemas so diferentes, i.e., se no dizem respeito ao 
    mesmo nmero ou se no tm a mesma durao.
    @pre V.
    @post operator!= = (um_telefonema.numero() != outro_telefonema.numero() ou
                        um_telefonema.duracao() != 
			outro_telefonema.duracao()). */
bool operator!=(Telefonema const& um_telefonema,
		Telefonema const& outro_telefonema);

/** Insere os dados de um telefonema num canal de sada.
    @pre V.
    @post canal_de_saida.fail() ou 
          (canal_de_saida.fail() e canal_de_saida contm o nmero e a 
	  durao de telefonema). */
std::ostream& operator<<(std::ostream& canal_de_saida, 
			 Telefonema const& telefonema);

/** Extrai os dados de um telefonema de um canal de entrada.
    @pre telefonema = t.
    @post (canal_de_entrada.fail() e telefonema = t) ou 
          (canal_de_entrada.fail() e telefonema.numero() e 
	   telefonema.duracao() contm a informao lida do canal e 
	   canal_de_entrada j no contm o nmero e a durao de telefonema 
	   extrados). */
std::istream& operator>>(std::istream& canal_de_entrada, 
			 Telefonema& telefonema);

#include "telefonema_impl.H"

#endif
