#include <cassert>

inline ListaDePonteiroAluno::Elo::Elo(Elo* const anterior, Elo* const seguinte,
				      Item const& item)
    : item(item), anterior(anterior), seguinte(seguinte)
{
}

inline ListaDePonteiroAluno::ListaDePonteiroAluno()
    : numero_de_itens(0),
      elo_inicial(new Elo),
      elo_final(new Elo(elo_inicial))
{
    elo_inicial->seguinte = elo_final;

    assert(cumpreInvariante());
}

inline ListaDePonteiroAluno::~ListaDePonteiroAluno()
{
    assert(cumpreInvariante());

    esvazia();

    delete elo_inicial;
    delete elo_final;
}

inline ListaDePonteiroAluno::Item const& ListaDePonteiroAluno::frente() const
{
    assert(cumpreInvariante());
    assert(not estaVazia());

    return elo_inicial->seguinte->item;
}

inline ListaDePonteiroAluno::Item const& ListaDePonteiroAluno::tras() const
{
    assert(cumpreInvariante());
    assert(not estaVazia());

    return elo_final->anterior->item;
}

inline int ListaDePonteiroAluno::comprimento() const
{
    assert(cumpreInvariante());

    return numero_de_itens;
}

inline bool ListaDePonteiroAluno::estaVazia() const
{
    assert(cumpreInvariante());

    return comprimento() == 0;
}

inline ListaDePonteiroAluno::Item& ListaDePonteiroAluno::frente()
{
    assert(cumpreInvariante());
    assert(not estaVazia());

    return elo_inicial->seguinte->item;
}

inline ListaDePonteiroAluno::Item& ListaDePonteiroAluno::tras()
{
    assert(cumpreInvariante());
    assert(not estaVazia());

    return elo_final->anterior->item;
}

/* Os mtodos poeNaFrente(), poeAtras() e insere() so todos
   implementados  custa do mtodo membro privado poe(): */

inline void ListaDePonteiroAluno::poeNaFrente(Item const& novo_item)
{
    assert(cumpreInvariante());

    poe(elo_inicial, elo_inicial->seguinte, novo_item);

    assert(cumpreInvariante());
}

inline void ListaDePonteiroAluno::poeAtras(Item const& novo_item)
{
    assert(cumpreInvariante());

    poe(elo_final->anterior, elo_final, novo_item);

    assert(cumpreInvariante());
}

inline void ListaDePonteiroAluno::insereAntes(Iterador const& iterador,
					      Item const& novo_item)
{
    assert(cumpreInvariante());
    assert(this == iterador.lista());
    assert(iterador != inicio());

    poe(iterador.elo()->anterior, iterador.elo(), novo_item);

    assert(cumpreInvariante());
}

/* Os mtodos tiraDaFrente(), tiraDeTras() e remove() so todos
   implementados  custa do mtodo privado tira(): */

inline void ListaDePonteiroAluno::tiraDaFrente()
{
    assert(cumpreInvariante());
    assert(not estaVazia());

    tira(elo_inicial->seguinte);

    assert(cumpreInvariante());
}

inline void ListaDePonteiroAluno::tiraDeTras()
{
    assert(cumpreInvariante());
    assert(not estaVazia());

    tira(elo_final->anterior);

    assert(cumpreInvariante());
}

inline void ListaDePonteiroAluno::remove(Iterador& iterador)
{
    assert(cumpreInvariante());
    assert(this == iterador.lista());
    assert(iterador != inicio());
    assert(iterador != fim());

    Elo* const elo_a_remover = iterador.elo();

    ++iterador;

    tira(elo_a_remover);

    assert(cumpreInvariante());
}

inline void ListaDePonteiroAluno::remove(Iterador const& iterador)
{
    assert(cumpreInvariante());
    assert(this == iterador.lista());
    assert(iterador != inicio());
    assert(iterador != fim());

    tira(iterador.elo());

    assert(cumpreInvariante());
}

inline
void ListaDePonteiroAluno::transfereDe(ListaDePonteiroAluno& outra_lista)
{
    assert(cumpreInvariante());
    assert(outra_lista.cumpreInvariante());

    /* Se forem a mesma lista transferir ... no fazer nada!  Se
       outra_lista estiver vazia tambm no  preciso fazer nada: */
    if(this != &outra_lista and not outra_lista.estaVazia()) {	
	// Basta trocar ponteiros!
	elo_final->anterior->seguinte = outra_lista.elo_inicial->seguinte;
	outra_lista.elo_inicial->seguinte->anterior = elo_final->anterior;
	outra_lista.elo_final->anterior->seguinte = elo_final;
	elo_final->anterior = outra_lista.elo_final->anterior;

	outra_lista.elo_inicial->seguinte = outra_lista.elo_final;
	outra_lista.elo_final->anterior = outra_lista.elo_inicial;
	
	numero_de_itens += outra_lista.numero_de_itens;
	outra_lista.numero_de_itens = 0;
    }

    assert(cumpreInvariante());
    assert(outra_lista.cumpreInvariante());
}

inline ListaDePonteiroAluno::Iterador ListaDePonteiroAluno::primeiro()
{
    assert(cumpreInvariante());

    return Iterador(*this);
}

inline ListaDePonteiroAluno::Iterador ListaDePonteiroAluno::ultimo()
{
    assert(cumpreInvariante());

    return Iterador(*this, elo_final->anterior);
}

inline ListaDePonteiroAluno::Iterador ListaDePonteiroAluno::inicio()
{
    assert(cumpreInvariante());

    return Iterador(*this, elo_inicial);
}

inline ListaDePonteiroAluno::Iterador ListaDePonteiroAluno::fim()
{
    assert(cumpreInvariante());

    return Iterador(*this, elo_final);
}

/* Coloca um novo elo, contendo o item novo_item, na cadeia duplamente
   ligada.  O novo elo  colocado entre o elo dado por anterior e o
   elo dado por seguinte. */
inline void ListaDePonteiroAluno::poe(Elo* const elo_anterior,
				      Elo* const elo_seguinte,
				      Item const& novo_item)
{
    assert(elo_anterior->seguinte == elo_seguinte);
    assert(elo_seguinte->anterior == elo_anterior);

    ++numero_de_itens;

    // Obter novo elo:
    Elo* const novo_elo = new Elo(elo_anterior, elo_seguinte, novo_item);

    elo_anterior->seguinte = novo_elo;
    elo_seguinte->anterior = novo_elo;
}

/* Retira o elo_a_tirar da cadeia duplamente ligada: */
inline void ListaDePonteiroAluno::tira(Elo* const elo_a_tirar)
{
    --numero_de_itens;

    elo_a_tirar->anterior->seguinte = elo_a_tirar->seguinte;
    elo_a_tirar->seguinte->anterior = elo_a_tirar->anterior;

    delete elo_a_tirar;
}

inline bool ListaDePonteiroAluno::cumpreInvariante() const
{
    if(numero_de_itens < 0)
	return false;
    Elo* elo_anterior = elo_inicial;
    Elo* elo = elo_inicial->seguinte;
    int i = 0;
    while(i != numero_de_itens and elo != elo_final) {
	if(elo->anterior != elo_anterior)
	    return false;
	elo_anterior = elo;
	elo = elo->seguinte;
	++i;
    }
    return elo->anterior == elo_anterior and
	elo == elo_final and i == numero_de_itens;
}


inline ListaDePonteiroAluno::Iterador::
Iterador(ListaDePonteiroAluno& lista_a_associar)
    : lista_associada(&lista_a_associar),
      elo_do_item_referenciado(lista_a_associar.elo_inicial->seguinte)
{
    assert(cumpreInvariante());
}

inline
ListaDePonteiroAluno::Item& ListaDePonteiroAluno::Iterador::operator*() const
{
    assert(cumpreInvariante());

    // O item s existe se o iterador no se referir a um item fictcio:
    assert(*this != lista_associada->inicio());
    assert(*this != lista_associada->fim());

    return elo_do_item_referenciado->item;
}

inline bool ListaDePonteiroAluno::Iterador::
operator==(Iterador const& outro_iterador) const
{
    assert(cumpreInvariante());
    assert(outro_iterador.cumpreInvariante());
    assert(lista_associada == outro_iterador.lista_associada);

    return elo_do_item_referenciado == outro_iterador.elo_do_item_referenciado;
}

inline bool ListaDePonteiroAluno::Iterador::
operator!=(Iterador const& outro_iterador) const
{
    return not(*this == outro_iterador);
}

inline
ListaDePonteiroAluno::Iterador& ListaDePonteiroAluno::Iterador::operator++()
{
    assert(cumpreInvariante());
    assert(*this != lista_associada->fim());

    elo_do_item_referenciado = elo_do_item_referenciado->seguinte;

    assert(cumpreInvariante());

    return *this;
}

inline
ListaDePonteiroAluno::Iterador& ListaDePonteiroAluno::Iterador::operator--()
{
    assert(cumpreInvariante());
    assert(*this != lista_associada->inicio());

    elo_do_item_referenciado = elo_do_item_referenciado->anterior;

    assert(cumpreInvariante());

    return *this;
}

inline
ListaDePonteiroAluno::Iterador ListaDePonteiroAluno::Iterador::operator++(int)
{
    assert(cumpreInvariante());

    ListaDePonteiroAluno::Iterador resultado = *this;
    ++*this;
;
    return resultado;
}

inline
ListaDePonteiroAluno::Iterador ListaDePonteiroAluno::Iterador::operator--(int)
{
    assert(cumpreInvariante());

    ListaDePonteiroAluno::Iterador resultado = *this;
    --*this;

    return resultado;
}

inline bool ListaDePonteiroAluno::Iterador::cumpreInvariante() const
{
    /* Em rigor dever-se-ia verificar se o elo pertence  cadeia dos
       elos, pois de outra forma o iterador est corrompido.  Mas a
       verdade  que os elos podem estar em estados invlidos... */
    return true /* simplificao grosseira. */;
}

inline ListaDePonteiroAluno::Iterador::
Iterador(ListaDePonteiroAluno& lista_a_associar,
	 Elo* const elo_do_item_a_referenciar)
    : lista_associada(&lista_a_associar),
      elo_do_item_referenciado(elo_do_item_a_referenciar)
{

    assert(cumpreInvariante());
}

inline
ListaDePonteiroAluno const* ListaDePonteiroAluno::Iterador::lista() const
{
    assert(cumpreInvariante());

    return lista_associada;
}

inline ListaDePonteiroAluno::Elo* ListaDePonteiroAluno::Iterador::elo() const
{
    assert(cumpreInvariante());

    return elo_do_item_referenciado;
}
