#ifndef LISTA_DE_DOUBLE_H
#define LISTA_DE_DOUBLE_H

#include <iostream>

/** @brief Representa listas de itens do tipo Item.

    Item  actualmente  um sinnimo de double, mas que pode ser
    alterado facilmente para o tipo que se entender.  Por conveno,
    chama-se "frente" e "tras" ao primeiro e ltimo item na lista.  Os
    nome "primeiro", "ltimo", "incio" e "fim" so reservados para
    iteradores.

    @invariant 0 <= numero_de_itens and cadeia duplamente ligada de
    elos tem numero_de_itens e  consistente. */
class ListaDeDouble {
  public:

    /** @brief Sinnimo de double.  Usa-se para simplificar a tarefa
	de criar listas com itens de outros tipos. */
    typedef double Item;

    /* Declarao de uma classe embutida que serve para percorrer e
       manipular listas: */
    class Iterador;

    /* Idem, mas garante que nem os itens nem a lista so alterados e
       por isso pode ser usada para percorrer listas constantes: */
    class IteradorConstante;


    // Construtores:

    /// Construtor da classe, cria uma lista vazia.
    ListaDeDouble();

    /// Construtor por cpia da classe, cria uma lista a partir de outra.
    ListaDeDouble(ListaDeDouble const& original);

    /// Destrutor da classe.
    ~ListaDeDouble();


    // Operador de atribuio por cpia:
    ListaDeDouble& operator=(ListaDeDouble modelo);


    // Inspectores:

    /** @brief Devolve referncia constante para o item na frente da lista.
	@pre estaVazia(). */
    Item const& frente() const;

    /** @brief Devolve referncia constante para o item na traseira da lista.
	@pre estaVazia(). */
    Item const& tras() const;

    /// Devolve o comprimento da lista, ou seja, o seu nmero de itens.
    int comprimento() const;

    /// Indica se a lista est vazia.
    bool estaVazia() const;

    /* Funes construtoras de iteradores constantes.  Consideram-se
       inspectoras porque a lista no pode ser modificada atravs dos
       iteradores. */

    /** @brief Devolve um novo primeiro iterador, i.e., um iterador
	referenciando o item na frente da lista.

	Note-se que se a lista estiver vazia o primeiro iterador 
	igual ao iterador final. */
    IteradorConstante primeiro() const;

    /** @brief Devolve um novo ultimo iterador, i.e., um iterador
	referenciando o item na traseira da lista.

	Note-se que se a lista estiver vazia o ltimo iterador  igual
	ao iterador inicial. */
    IteradorConstante ultimo() const;

    /** @brief Devolve um novo iterador inicial, i.e., um iterador
	referenciando o item fictcio imediatamente antes do item na
	frente da lista. */
    IteradorConstante inicio() const;

    /** @brief Devolve um novo iterador final, i.e., um iterador
	referenciando o item fictcio imediatamente aps o item na
	traseira da lista. */
    IteradorConstante fim() const;

    /** @brief Procura a primeira ocorrncia de um dado item.

	Devolve iterador referenciando o item de trs se o item
	procurado no existir.

	@see ultimaOcorrenciaDe(). */
    IteradorConstante primeiraOcorrenciaDe(Item const& item) const;

    /** @brief Procura a ultimo ocorrncia de um dado item.

	Devolve iterador referenciando o item da frente se o item
	procurado no existir.

	@see primeiraOcorrenciaDe(). */
    IteradorConstante ultimaOcorrenciaDe(Item const& item) const;


    // Modificadores:

    /** @brief Devolve referncia para o item na frente da lista.

	No modifica directamente a lista, mas permite modificaes
	atravs da referncia devolvida.

	@pre estaVazia(). */
    Item& frente();

    /** @brief Devolve referncia para o item na traseira da lista.
	@pre estaVazia(). */
    Item& tras();

    /** @brief Poe novo item na frente da lista.
	@pre V.
	@post H um novo item na frente da lista ou a excepo
	bad_alloc foi lanada. */
    void poeNaFrente(Item const& novo_item);

    /** @brief Poe novo item na traseira da lista.
	@pre V.
	@post H um novo item na traseira da lista ou a excepo
	bad_alloc foi lanada. */
    void poeAtras(Item const& novo_item);

    /** @brief Insere novo item imediatamente antes da posio indicada pelo
	iterador i.

	Faz com que o iterador continue a referenciar o mesmo item que
	antes da insero.

	@pre iterador  vlido e iterador <> fim().
	@post H um novo item antes do referenciado pelo iterador ou a
	excepo bad_alloc foi lanada. */
    void insereAntes(Iterador const& iterador, Item const& novo_item);

    /** @brief Insere novo item imediatamente antes da posio indicada pelo
	iterador i, que  um iterador constante.

	Faz com que o iterador continue a referenciar o mesmo item que
	antes da insero.

	@pre iterador  vlido e iterador <> fim().
	@post H um novo item antes do referenciado pelo iterador ou a
	excepo bad_alloc foi lanada. */
    void insereAntes(IteradorConstante const& iterador, Item const& novo_item);

    /** @brief Tira o item da frente da lista.
	@pre estaVazia(). */
    void tiraDaFrente();

    /** @brief Tira o item da traseira da lista.
	@pre estaVazia(). */
    void tiraDeTras();

    /// Esvazia a lista.
    void esvazia();

    /** @brief Remove o item indicado pelo iterador i.

	O iterador fica a referenciar o item logo aps o item
	removido.

	@pre iterador  vlido e iterador <> inicio() e iterador <> fim(). */
    void remove(Iterador& iterador);

    /** @brief Remove o item indicado pelo iterador i, que  um iterador
	constante.

	O iterador fica a referenciar o item logo aps o item
	removido.

	@pre iterador  vlido e iterador <> inicio() e iterador <> fim(). */
    void remove(IteradorConstante& iterador);

    /** @brief Remove o item indicado pelo iterador i que fica invlido.

	O iterador  invalidado pela operao!

	@pre iterador  vlido e iterador <> inicio() e iterador <> fim().
	@see remove(Iterador&). */
    void remove(Iterador const& iterador);

    /** @brief Remove o item indicado pelo iterador i que fica invlido.

	O iterador i  constante e  invalidado pela operao!

	@pre iterador  vlido e iterador <> inicio() e iterador <> fim().
	@see remove(Iterador&). */
    void remove(IteradorConstante const& iterador);

    /** @brief Transfere todos itens de lista_fonte para a lista
	implcita.

	A lista_fonte fica vazia!

	@see operator+=(). */
    void transfereDe(ListaDeDouble& lista_fonte);

    /** @brief Troca todos itens de outra_lista para a lista implcita
	e vice-versa.

	@see transfere(). */
    void trocaCom(ListaDeDouble& outra_lista);

    /** @brief Concatena a lista_fonte com a lista implcita.

	A lista_fonte no sofre qualquer alterao.

	@see transfere(). */
    ListaDeDouble& operator+=(ListaDeDouble lista_fonte);


    /* Funes construtoras de iteradores.  Consideram-se
       modificadoras porque a lista pode ser modificada atravs dos
       iteradores. */

    /** @brief Devolve um novo primeiro iterador, i.e., um iterador
	referenciando o item na frente da lista.

	Note-se que se a lista estiver vazia o primeiro iterador 
	igual ao iterador final. */
    Iterador primeiro();

    /** @brief Devolve um novo ultimo iterador, i.e., um iterador
	referenciando o item na traseira da lista.

	Note-se que se a lista estiver vazia o ltimo iterador  igual
	ao iterador inicial. */
    Iterador ultimo();

    /** @brief Devolve um novo iterador inicial, i.e., um iterador
	referenciando o item fictcio imediatamente antes do item na
	frente da lista. */
    Iterador inicio();

    /** @brief Devolve um novo iterador final, i.e., um iterador
	referenciando o item fictcio imediatamente aps o item na
	traseira da lista. */
    Iterador fim();

    /** @brief Procura a primeira ocorrncia de um dado item.

	Devolve iterador referenciando o item de trs se o item procurado no
	existir.

	@see ultimaOcorrenciaDe(). */
    Iterador primeiraOcorrenciaDe(Item const& item);

    /** @brief Procura a ultimo ocorrncia de um dado item.

	Devolve iterador referenciando o item da frente se o item
	procurado no existir.

	@see primeiraOcorrenciaDe(). */
    Iterador ultimaOcorrenciaDe(Item const& item);

  private:

    /* A lista  representada por uma cadeia duplamente ligada de elos
       guardados.  A cadeia duplamente ligada tem guardas
       (correspondentes aos itens fictcios): */
    struct Elo {
	Elo(Elo* const anterior = 0, Elo* const seguinte = 0,
	    Item const& item = Item());

	// O item propriamente dito:
	Item item;
	// Ponteiro para o elo anterior na cadeia:
	Elo* anterior;
	// Ponteiro para o elo seguinte na cadeia:
	Elo* seguinte;
    };

    // Contador do nmero de itens na lista:
    int numero_de_itens;

    // Ponteiros para as guardas da cadeia duplamente ligada de itens:
    Elo* elo_inicial;
    Elo* elo_final;

    /* Procedimentos auxiliares para inserir e remover um item da
       cadeia duplamente ligada dos itens: */
    void poe(Elo* const elo_anterior, Elo* const elo_seguinte,
	     Item const& novo_item);
    void tira(Elo* const elo_a_tirar);

    /* Funo auxiliar que indica se a condio invariante de
       instncia da classe se verifica: */
    bool cumpreInvariante() const;

    // As classes de iterao tm acesso irrestrito s listas:
    friend class Iterador;
    friend class IteradorConstante;
};


/** @brief Representa iteradores para itens de listas do tipo
    ListaDeDouble.

    Os iteradores tm uma caracterstica infeliz: podem estar em
    estados invlidos.	Por exemplo, se uma lista for esvaziada, todos
    os iteradores a ela associada ficam invlidos.   possvel
    resolver este problema, mas  custa de um aumento considervel da
    complexidade deste par de classes.

    @invariant V (uma simplificao grosseira). */
class ListaDeDouble::Iterador {
  public:

    // Construtores:

    /** @brief Construtor da classe.
	
	Associa o iterador com a lista passada como argumento e pe-no
	a referenciar o item na sua frente. */
    explicit Iterador(ListaDeDouble& lista_a_associar);


    // Inspectores:

    /** @brief Devolve uma referncia para o item referenciado pelo
	iterador.

	Note-se que a referncia devolvida no  constante.   que um
	iterador const no pode ser alterado (avanar ou recuar), mas
	permite alerar o item por ele referenciado na lista
	associada.

	@pre O item referenciado no pode ser nenhum dos itens
	fictcios da lista (i.e., nem o item antes da frente da lista,
	nem o item aps a sua traseira). */
    Item& operator*() const;

    /** @brief Indica se dois iteradores so iguais.

	Ou melhor, se a instncia implcita  igual ao iterador passado como
	argumento.  Dois iteradores so iguais se se referirem ao mesmo item
	da mesma lista (mesmo que sejam itens fictcios). A comparao s faz
	sentido se os iteradores estiverem associados  mesma lista! */
    bool operator==(Iterador const& outro_iterador) const;

    /// Operador de diferena entre iteradores.
    bool operator!=(Iterador const& outro_iterador) const;

    /// Operador de igualdade entre um iterador e um iterador constante.
    bool operator==(IteradorConstante const& outro_iterador) const;

    /// Operador de diferena entre um iterador e um iterador constante.
    bool operator!=(IteradorConstante const& outro_iterador) const;


    // Modificadores:

    /** @brief Avana iterador para o prximo item da lista.  Devolve o
	prprio iterador.
	@pre O iterador no pode ser o iterador final da lista. */
    Iterador& operator++();

    /** @brief Recua iterador para o item anterior da lista.  Devolve o
	prprio iterador.
	@pre O iterador no pode ser o iterador inicial da lista. */
    Iterador& operator--();

    /** @brief Avana iterador para o prximo item da lista.  Devolve um novo
	iterador com o valor do prprio iterador antes de avanado.
	@pre O iterador no pode ser o iterador final da lista. */
    Iterador operator++(int);

    /** @brief Recua iterador para o item anterior da lista.  Devolve um novo
	iterador com o valor do prprio iterador antes de recuado.
	@pre O iterador no pode ser o iterador inicial da lista. */
    Iterador operator--(int);


  private:

    // Ponteiro para a lista a que o iterador est associado:
    ListaDeDouble* lista_associada;

    // Ponteiro para o elo do item da lista referenciado pelo iterador:
    Elo* elo_do_item_referenciado;

    /* Construtor privado da classe.  Associa o iterador com a
       lista_a_associar e pe-no a referenciar o item no elo
       indicado. */
    Iterador(ListaDeDouble& lista_a_associar,
	     Elo* const elo_do_item_a_referenciar);

    /* Inspector para obter o ponteiro para o elo do item referenciado pelo
       iterador: */
    Elo* elo() const;

    // Inspector para obter a lista associada ao iterador:
    ListaDeDouble const* lista() const;

    // Funo auxiliar que indica se a condio invariante de
    // instncia da classe se verifica:
    bool cumpreInvariante() const;

    /* A classe IteradorConstante  amiga da classe Iterador, para que o
       construtor da classe IteradorConstante por cpia de um Iterador possa
       aceder s operaes privadas de Iterador. */
    friend class IteradorConstante;

    /* A classe ListaDeDouble tem acesso irrestrito a todos os membros
       da classe Iterador.   importante perceber que as duas classes,
       ListaDeDouble e ListaDeDouble::Iterador esto completamente
       interligadas.  No h qualquer promiscuidade nesta relao.
       So partes do mesmo todo. */
    friend class ListaDeDouble;
};


/** @brief Representa iteradores para itens constantes de listas
    constantes do tipo ListaDeDouble.  Estes iteradores no permitem
    alterar os itens referenciados nem as listas associadas.

    @invariant V (uma simplificao grosseira). */
class ListaDeDouble::IteradorConstante {
  public:

    // Construtores:

    /** @brief Construtor da classe.

	Associa o iterador com a lista passada como argumento e pe-no
	a referenciar o item na sua frente. */
    explicit IteradorConstante(ListaDeDouble const& lista_a_associar);

    /** @brief Construtor por cpia a partir de iterador normal.

	Define converso implcita de um iterador no constante num
	iterador constante. */
    IteradorConstante(Iterador const& outro_iterador);


    // Inspectores:

    /** @brief Devolve uma referncia para o item referenciado pelo
	iterador.

	Note-se que a referncia devolvida  constante.	  que um iterador
	constante que seja const no pode ser alterado (avanar ou recuar),
	new permite alterar o item por ele referenciado na lista associada.

	@pre O item referenciado no pode ser nenhum dos itens
	fictcios da lista (i.e., nem o item antes da frente da lista,
	nem o item aps a sua traseira). */
    Item const& operator*() const;

    /** @brief Indica se dois iteradores so iguais.

	Ou melhor, se a instncia implcita  igual ao iterador passado como
	argumento.  Dois iteradores so iguais se se referirem ao mesmo item
	da mesma lista (mesmo que sejam itens fictcios). A comparao s faz
	sentido se os iteradores estiverem associados  mesma lista! */
    bool operator==(IteradorConstante const& outro_iterador) const;

    /// Operador de diferena entre iteradores.
    bool operator!=(IteradorConstante const& outro_iterador) const;


    // Modificadores:

    /** @brief Avana iterador para o prximo item da lista.  Devolve o
	prprio iterador.
	@pre O iterador no pode ser o iterador final da lista. */
    IteradorConstante& operator++();

    /** @brief Recua iterador para o item anterior da lista.  Devolve o
	prprio iterador.
	@pre O iterador no pode ser o iterador inicial da lista. */
    IteradorConstante& operator--();

    /** @brief Avana iterador para o prximo item da lista.  Devolve um novo
	iterador com o valor do prprio iterador antes de avanado.
	@pre O iterador no pode ser o iterador final da lista. */
    IteradorConstante operator++(int);

    /** @brief Recua iterador para o item anterior da lista.  Devolve um novo
	iterador com o valor do prprio iterador antes de recuado.
	@pre O iterador no pode ser o iterador inicial da lista. */
    IteradorConstante operator--(int);


  private:

    // Ponteiro para a lista a que o iterador est associado:
    ListaDeDouble const* lista_associada;

    // Ponteiro para o elo do item da lista referenciado pelo iterador:
    Elo* elo_do_item_referenciado;

    /* Repare-se que, estranhamente, se guarda um ponteiro para um elo
       no-constante.  A garantia de constncia  data pelo programador, e no
       pelo compilador.	 Desta forma resolve-se o problema de ser ou no
       vlido inserir ou remover numa ou de uma lista no-constante dado um
       iterador constante.  A opo foi a que nos pareceu mais razovel.  Se a
       lista  no-constante, ento pode ser alterada.	O facto do iterador
       no permitir alteraes da lista deixa de ter influncia.  Para
       simplificaro o cdigo (i.e., torn-lo mais eficiente ou evitar recorrer
       a converses para remoo do qualificador const, adoptou-se esta
       soluo. */

    /* Construtor privado da classe.  Associa o iterador com a
       lista_a_associar e pe-no a referenciar o item no elo
       indicado. */
    IteradorConstante(ListaDeDouble const& lista_a_associar,
		      Elo* elo_do_item_a_referenciar);

    /* Inspector para obter o ponteiro para o elo do item referenciado pelo
       iterador: */
    Elo* elo() const;

    // Inspector para obter a lista associada ao iterador:
    ListaDeDouble const* lista() const;

    // Funo auxiliar que indica se a condio invariante de
    // instncia da classe se verifica:
    bool cumpreInvariante() const;

    /* A classe Iterador  amiga da classe IteradorConstante, para que o
       operador == da classe Iterador possa aceder aos mtodos privados de
       IteradorConstante. */
    friend class Iterador;

    /* A classe ListaDeDouble tem acesso irrestrito a todos os membros da classe
       IteradorConstante.   importante perceber que as duas classes,
       ListaDeDouble e ListaDeDouble::Iterador esto completamente interligadas.
       No h qualquer promiscuidade nesta relao.  So partes do mesmo
       todo. */
    friend class ListaDeDouble;
};


/** @brief Operador de insero de listas num canal.

    Todos os itens da lista so inseridos por ordem entre parnteses e
    separados por vrgulas.

    Usa-se ListaDeDouble& em vez de ListaDeDouble const&, porque caso
    constrrio ter-se-ia de definir uma classe de iterador para listas
    constantes (coisa que ser feita mais tarde). */
std::ostream& operator<<(std::ostream& saida, ListaDeDouble const& lista);


#include "lista_de_double_impl.H"

#endif // LISTA_DE_DOUBLE_H
