#include <cassert>
#include <algorithm>

inline ListaDeDouble::Elo::Elo(Elo* const anterior, Elo* const seguinte,
			       Item const& item)
    : item(item), anterior(anterior), seguinte(seguinte)
{
}

inline ListaDeDouble::ListaDeDouble()
    : numero_de_itens(0),
      elo_inicial(new Elo)
{
    // Captura-se qualquer excepo, pois reimplementaes podem levar
    // o construtor dos itens a lanar outro tipo de excepo que no
    // bad_alloc.
    try {
	elo_inicial->seguinte = elo_final = new Elo(elo_inicial);
    } catch(...) {
	// Captura-se qualquer excepo lanada.

	// Destri-se o elo inicial entretanto criado sem problemas.
	delete elo_inicial;

	// Finalmente  necessrio relanar a excepo!	 Afinal, houve
	// um erro e por isso a construo falhou...
	throw;
    }

    assert(cumpreInvariante());
}

inline ListaDeDouble::~ListaDeDouble()
{
    assert(cumpreInvariante());

    esvazia();

    delete elo_inicial;
    delete elo_final;
}

/* O problema aqui  fazer com que isto funcione apesar de possveis
   excepes.  Podem ocorrer excepes:

   1.  Ao construir os elos.

   2.  Ao construir os itens nos elos.	Se Item for o mesmo que int, isto
   no acontece, mas ns queremos poder mudar Item para outro tipo
   qualquer, que pode lanar uma excepo ao ser construdo...

   3.  Ao atribuir os itens nos elos, se se reciclarem os elos.	 Mais uma
   vez com Item sendo int, isto no acontece, mas ns queremos poder mudar
   Item  vontade...

   Se alguma coisa falhar, ento considera-se que toda a operao de
   atribuio falhou e portanto a lista implcita deve ficar intacta, tal
   como estava no incio...  Isso implica que no se pode reciclar, pois
   reciclar implica perder os valores dos itens...

   A melhor soluo (proposta por Stroustrup)  simples.  Construir uma
   nova lista por cpia de modelo (o construtor por cpia comporta-se
   bem em caso de erro) e depois trocar os elos dessa cpia com os da
   lista implcita.  Quando o operador terminar, a cpia  destruda
   junto com todos os elos que estavam na lista implcita originalmente.
   Para simplificar a soluo enriqueceu-se a interface da classe com um
   procedimento membro trocaCom() que troca os itens de duas listas.

inline ListaDeDouble& ListaDouble::operator=(ListaDeDouble const& modelo)
{
    assert(cumpreInvariante());
    assert(modelo.cumpreInvariante());


    if(this != &modelo) {
	ListaDeDouble copia_da_modelo(modelo);
	
	// Se a copia falhar a troca no chega a ser feita.  Logo, a instncia
	// implcita no  alterada!
	trocaCom(copia_da_modelo);
    }
    // Explique porque  dispensvel o habitual teste if(this != &modelo)...

    return *this;
}

   Ou mais simples ainda: */

inline ListaDeDouble& ListaDeDouble::operator=(ListaDeDouble copia_de_modelo)
{
    assert(cumpreInvariante());
    assert(copia_de_modelo.cumpreInvariante());

    trocaCom(copia_de_modelo);

    assert(cumpreInvariante());
    assert(copia_de_modelo.cumpreInvariante());

    return *this;
}

inline ListaDeDouble::Item const& ListaDeDouble::frente() const
{
    assert(cumpreInvariante());
    assert(not estaVazia());

    return elo_inicial->seguinte->item;
}

inline ListaDeDouble::Item const& ListaDeDouble::tras() const
{
    assert(cumpreInvariante());
    assert(not estaVazia());

    return elo_final->anterior->item;
}

inline int ListaDeDouble::comprimento() const
{
    assert(cumpreInvariante());

    return numero_de_itens;
}

inline bool ListaDeDouble::estaVazia() const
{
    assert(cumpreInvariante());

    return comprimento() == 0;
}

inline ListaDeDouble::IteradorConstante ListaDeDouble::primeiro() const
{
    assert(cumpreInvariante());

    return IteradorConstante(*this);
}

inline ListaDeDouble::IteradorConstante ListaDeDouble::ultimo() const
{
    assert(cumpreInvariante());

    return IteradorConstante(*this, elo_final->anterior);
}

inline ListaDeDouble::IteradorConstante ListaDeDouble::inicio() const
{
    assert(cumpreInvariante());

    return IteradorConstante(*this, elo_inicial);
}

inline ListaDeDouble::IteradorConstante ListaDeDouble::fim() const
{
    assert(cumpreInvariante());

    return IteradorConstante(*this, elo_final);
}

inline ListaDeDouble::Item& ListaDeDouble::frente()
{
    assert(cumpreInvariante());
    assert(not estaVazia());

    return elo_inicial->seguinte->item;
}

inline ListaDeDouble::Item& ListaDeDouble::tras()
{
    assert(cumpreInvariante());
    assert(not estaVazia());

    return elo_final->anterior->item;
}

/* Os mtodos poeNaFrente(), poeAtras() e insere() so todos
   implementados  custa do mtodo membro privado poe(): */

inline void ListaDeDouble::poeNaFrente(Item const& novo_item)
{
    assert(cumpreInvariante());

    poe(elo_inicial, elo_inicial->seguinte, novo_item);

    assert(cumpreInvariante());
}

inline void ListaDeDouble::poeAtras(Item const& novo_item)
{
    assert(cumpreInvariante());

    poe(elo_final->anterior, elo_final, novo_item);

    assert(cumpreInvariante());
}

inline void ListaDeDouble::insereAntes(Iterador const& iterador,
				      Item const& novo_item)
{
    assert(cumpreInvariante());
    assert(this == iterador.lista());
    assert(iterador != inicio());

    poe(iterador.elo()->anterior, iterador.elo(), novo_item);

    assert(cumpreInvariante());
}

inline void ListaDeDouble::insereAntes(IteradorConstante const& iterador,
				      Item const& novo_item)
{
    assert(cumpreInvariante());
    assert(this == iterador.lista());
    assert(iterador != inicio());

    poe(iterador.elo()->anterior, iterador.elo(), novo_item);

    assert(cumpreInvariante());
}

/* Os mtodos tiraDaFrente(), tiraDeTras() e remove() so todos
   implementados  custa do mtodo privado tira(): */

inline void ListaDeDouble::tiraDaFrente()
{
    assert(cumpreInvariante());
    assert(not estaVazia());

    tira(elo_inicial->seguinte);

    assert(cumpreInvariante());
}

inline void ListaDeDouble::tiraDeTras()
{
    assert(cumpreInvariante());
    assert(not estaVazia());

    tira(elo_final->anterior);

    assert(cumpreInvariante());
}

inline void ListaDeDouble::remove(Iterador& iterador)
{
    assert(cumpreInvariante());
    assert(this == iterador.lista());
    assert(iterador != inicio());
    assert(iterador != fim());

    Elo* const elo_a_remover = iterador.elo();

    ++iterador;

    tira(elo_a_remover);

    assert(cumpreInvariante());
}

inline void ListaDeDouble::remove(IteradorConstante& iterador)
{
    assert(cumpreInvariante());
    assert(this == iterador.lista());
    assert(iterador != inicio());
    assert(iterador != fim());

    Elo* const elo_a_tirar = iterador.elo();

    ++iterador;

    tira(elo_a_tirar);

    assert(cumpreInvariante());
}

inline void ListaDeDouble::remove(Iterador const& iterador)
{
    assert(cumpreInvariante());
    assert(this == iterador.lista());
    assert(iterador != inicio());
    assert(iterador != fim());

    tira(iterador.elo());

    assert(cumpreInvariante());
}

inline void ListaDeDouble::remove(IteradorConstante const& iterador)
{
    assert(cumpreInvariante());
    assert(this == iterador.lista());
    assert(iterador != inicio());
    assert(iterador != fim());

    tira(iterador.elo());

    assert(cumpreInvariante());
}

inline void ListaDeDouble::transfereDe(ListaDeDouble& lista_fonte)
{
    assert(cumpreInvariante());
    assert(lista_fonte.cumpreInvariante());

    /* Se forem a mesma lista transferir ... no fazer nada!  Se
       lista_fonte estiver vazia tambm no  preciso fazer nada: */
    if(this != &lista_fonte and not lista_fonte.estaVazia()) {	
	// Basta trocar ponteiros!
	elo_final->anterior->seguinte = lista_fonte.elo_inicial->seguinte;
	lista_fonte.elo_inicial->seguinte->anterior = elo_final->anterior;
	lista_fonte.elo_final->anterior->seguinte = elo_final;
	elo_final->anterior = lista_fonte.elo_final->anterior;

	lista_fonte.elo_inicial->seguinte = lista_fonte.elo_final;
	lista_fonte.elo_final->anterior = lista_fonte.elo_inicial;
	
	numero_de_itens += lista_fonte.numero_de_itens;
	lista_fonte.numero_de_itens = 0;
    }

    assert(cumpreInvariante());
    assert(lista_fonte.cumpreInvariante());
}

inline void ListaDeDouble::trocaCom(ListaDeDouble& outra_lista)
{
    assert(cumpreInvariante());
    assert(outra_lista.cumpreInvariante());

    // Este procedimento std::swap() troca o valor de duas variveis do mesmo
    // tipo.  Est declarado em #include <algorithm>:
    std::swap(elo_inicial, outra_lista.elo_inicial);
    std::swap(elo_final, outra_lista.elo_final);
    std::swap(numero_de_itens, outra_lista.numero_de_itens);

    assert(cumpreInvariante());
    assert(outra_lista.cumpreInvariante());
}

inline ListaDeDouble& ListaDeDouble::operator+=(ListaDeDouble lista_fonte)
{
    assert(cumpreInvariante());
    assert(lista_fonte.cumpreInvariante());

    transfereDe(lista_fonte);

    assert(cumpreInvariante());

    return *this;
}

inline ListaDeDouble::Iterador ListaDeDouble::primeiro()
{
    assert(cumpreInvariante());

    return Iterador(*this);
}

inline ListaDeDouble::Iterador ListaDeDouble::ultimo()
{
    assert(cumpreInvariante());

    return Iterador(*this, elo_final->anterior);
}

inline ListaDeDouble::Iterador ListaDeDouble::inicio()
{
    assert(cumpreInvariante());

    return Iterador(*this, elo_inicial);
}

inline ListaDeDouble::Iterador ListaDeDouble::fim()
{
    assert(cumpreInvariante());

    return Iterador(*this, elo_final);
}

/* Coloca um novo elo, contendo o item novo_item, na cadeia duplamente
   ligada.  O novo elo  colocado entre o elo dado por anterior e o
   elo dado por seguinte. */
inline void ListaDeDouble::poe(Elo* const elo_anterior, Elo* const elo_seguinte,
			      Item const& novo_item)
{
    assert(elo_anterior->seguinte == elo_seguinte);
    assert(elo_seguinte->anterior == elo_anterior);

    // Obter novo elo:
    Elo* const novo_elo = new Elo(elo_anterior, elo_seguinte, novo_item);

    ++numero_de_itens;

    elo_anterior->seguinte = novo_elo;
    elo_seguinte->anterior = novo_elo;
}

inline void ListaDeDouble::tira(Elo* const elo_a_tirar)
{
    --numero_de_itens;

    elo_a_tirar->anterior->seguinte = elo_a_tirar->seguinte;
    elo_a_tirar->seguinte->anterior = elo_a_tirar->anterior;

    delete elo_a_tirar;
}

inline bool ListaDeDouble::cumpreInvariante() const
{
    if(numero_de_itens < 0)
	return false;
    Elo* elo_anterior = elo_inicial;
    Elo* elo = elo_inicial->seguinte;
    int i = 0;
    while(i != numero_de_itens and elo != elo_final) {
	if(elo->anterior != elo_anterior)
	    return false;
	elo_anterior = elo;
	elo = elo->seguinte;
	++i;
    }
    return elo->anterior == elo_anterior and
	elo == elo_final and i == numero_de_itens;
}


inline ListaDeDouble::Iterador::Iterador(ListaDeDouble& lista_a_associar)
    : lista_associada(&lista_a_associar),
      elo_do_item_referenciado(lista_a_associar.elo_inicial->seguinte)
{
    assert(cumpreInvariante());
}

inline ListaDeDouble::Item& ListaDeDouble::Iterador::operator*() const
{
    assert(cumpreInvariante());
    assert(*this != lista_associada->inicio());
    assert(*this != lista_associada->fim());

    return elo_do_item_referenciado->item;
}

inline bool
ListaDeDouble::Iterador::operator==(Iterador const& outro_iterador) const
{
    assert(cumpreInvariante());
    assert(outro_iterador.cumpreInvariante());
    assert(lista_associada == outro_iterador.lista_associada);

    return elo_do_item_referenciado == outro_iterador.elo_do_item_referenciado;
}

inline bool
ListaDeDouble::Iterador::operator!=(Iterador const& outro_iterador) const
{
    assert(cumpreInvariante());
    assert(outro_iterador.cumpreInvariante());
    assert(lista_associada == outro_iterador.lista_associada);

    return not (*this == outro_iterador);
}

inline bool ListaDeDouble::Iterador::
operator==(IteradorConstante const& outro_iterador) const
{
    assert(cumpreInvariante());
    assert(outro_iterador.cumpreInvariante());
    assert(lista_associada == outro_iterador.lista_associada);

    return elo_do_item_referenciado == outro_iterador.elo_do_item_referenciado;
}

inline bool ListaDeDouble::Iterador::
operator!=(IteradorConstante const& outro_iterador) const
{
    assert(cumpreInvariante());
    assert(outro_iterador.cumpreInvariante());
    assert(lista_associada == outro_iterador.lista_associada);

    return not (*this == outro_iterador);
}

inline ListaDeDouble::Iterador& ListaDeDouble::Iterador::operator++()
{
    assert(cumpreInvariante());
    assert(*this != lista_associada->fim());

    elo_do_item_referenciado = elo_do_item_referenciado->seguinte;

    assert(cumpreInvariante());

    return *this;
}

inline ListaDeDouble::Iterador& ListaDeDouble::Iterador::operator--()
{
    assert(cumpreInvariante());
    assert(*this != lista_associada->inicio());

    elo_do_item_referenciado = elo_do_item_referenciado->anterior;

    assert(cumpreInvariante());

    return *this;
}

inline ListaDeDouble::Iterador ListaDeDouble::Iterador::operator++(int)
{
    assert(cumpreInvariante());

    ListaDeDouble::Iterador resultado = *this;
    ++*this;

    assert(cumpreInvariante());

    return resultado;
}

inline ListaDeDouble::Iterador ListaDeDouble::Iterador::operator--(int)
{
    assert(cumpreInvariante());

    ListaDeDouble::Iterador resultado = *this;
    --*this;

    assert(cumpreInvariante());

    return resultado;
}

inline bool ListaDeDouble::Iterador::cumpreInvariante() const
{
    /* Em rigor dever-se-ia verificar se o elo pertence  cadeia dos
       elos, pois de outra forma o iterador est corrompido.  Mas a
       verdade  que os elos podem estar em estados invlidos... */
    return true /* simplificao grosseira. */;
}

inline ListaDeDouble::Iterador::Iterador(ListaDeDouble& lista_a_associar,
					Elo* const elo_do_item_a_referenciar)
    : lista_associada(&lista_a_associar),
      elo_do_item_referenciado(elo_do_item_a_referenciar)
{
    assert(cumpreInvariante());
}

inline ListaDeDouble const* ListaDeDouble::Iterador::lista() const
{
    assert(cumpreInvariante());

    return lista_associada;
}

inline ListaDeDouble::Elo* ListaDeDouble::Iterador::elo() const
{
    assert(cumpreInvariante());

    return elo_do_item_referenciado;
}

inline ListaDeDouble::IteradorConstante::
IteradorConstante(ListaDeDouble const& lista_a_associar)
    : lista_associada(&lista_a_associar),
      elo_do_item_referenciado(lista_a_associar.elo_inicial->seguinte)
{
    assert(cumpreInvariante());
}

inline ListaDeDouble::IteradorConstante::
IteradorConstante(Iterador const& outro_iterador)
    : lista_associada(outro_iterador.lista()),
      elo_do_item_referenciado(outro_iterador.elo())
{
    assert(cumpreInvariante());
}

inline ListaDeDouble::Item const& ListaDeDouble::IteradorConstante::
operator*() const
{
    assert(cumpreInvariante());
    assert(*this != lista_associada->inicio());
    assert(*this != lista_associada->fim());

    return elo_do_item_referenciado->item;
}

inline bool ListaDeDouble::IteradorConstante::
operator==(IteradorConstante const& outro_iterador) const
{
    assert(cumpreInvariante());
    assert(outro_iterador.cumpreInvariante());
    assert(lista_associada == outro_iterador.lista_associada);

    return elo_do_item_referenciado == outro_iterador.elo_do_item_referenciado;
}

inline bool ListaDeDouble::IteradorConstante::
operator!=(IteradorConstante const& outro_iterador) const
{
    return not (*this == outro_iterador);
}

inline ListaDeDouble::IteradorConstante& ListaDeDouble::IteradorConstante::
operator++()
{
    assert(cumpreInvariante());
    assert(*this != lista_associada->fim());

    elo_do_item_referenciado = elo_do_item_referenciado->seguinte;

    assert(cumpreInvariante());

    return *this;
}

inline ListaDeDouble::IteradorConstante& ListaDeDouble::IteradorConstante::
operator--()
{
    assert(cumpreInvariante());
    assert(*this != lista_associada->inicio());

    elo_do_item_referenciado = elo_do_item_referenciado->anterior;

    assert(cumpreInvariante());

    return *this;
}

inline ListaDeDouble::IteradorConstante ListaDeDouble::IteradorConstante::
operator++(int)
{
    assert(cumpreInvariante());

    ListaDeDouble::IteradorConstante resultado = *this;
    ++*this;

    assert(cumpreInvariante());

    return resultado;
}

inline ListaDeDouble::IteradorConstante
ListaDeDouble::IteradorConstante::operator--(int) {
    assert(cumpreInvariante());

    ListaDeDouble::IteradorConstante resultado = *this;
    --*this;

    assert(cumpreInvariante());

    return resultado;
}

inline bool ListaDeDouble::IteradorConstante::cumpreInvariante() const
{
    /* Em rigor dever-se-ia verificar se o elo pertence  cadeia dos
       elos, pois de outra forma o iterador est corrompido.  Mas a verdade 
       que os elos podem estar em estados invlidos... */
    return true /* simplificao grosseira. */;
}

inline ListaDeDouble::IteradorConstante::
IteradorConstante(ListaDeDouble const& lista_a_associar,
		  Elo* const elo_do_item_a_referenciar)
    : lista_associada(&lista_a_associar),
      elo_do_item_referenciado(elo_do_item_a_referenciar)
{
    assert(cumpreInvariante());
}

inline ListaDeDouble const* ListaDeDouble::IteradorConstante::lista() const
{
    assert(cumpreInvariante());

    return lista_associada;
}

inline ListaDeDouble::Elo* ListaDeDouble::IteradorConstante::elo() const
{
    assert(cumpreInvariante());

    return elo_do_item_referenciado;
}
