#include <cassert>
#include <algorithm>

inline ListaDeInt::Elo::Elo(Elo* const anterior, Elo* const seguinte,
			    Item const& item)
    : item(item), anterior(anterior), seguinte(seguinte)
{
}

inline ListaDeInt::ListaDeInt()
    : numero_de_itens(0),
    elo_inicial(new Elo)
{
    // Captura-se qualquer excepo, pois reimplementaes podem levar
    // o construtor dos itens a lanar outro tipo de excepo que no
    // bad_alloc.
    try {
	elo_inicial->seguinte = elo_final = new Elo(elo_inicial);
    } catch(...) {
	// Captura-se qualquer excepo lanada.

	// Destri-se o elo inicial entretanto criado sem problemas.
	delete elo_inicial;

	// Finalmente  necessrio relanar a excepo!	 Afinal, houve
	// um erro e por isso a construo falhou...
	throw;
    }

    assert(cumpreInvariante());
}

inline ListaDeInt::~ListaDeInt()
{
    assert(cumpreInvariante());

    esvazia();

    delete elo_inicial;
    delete elo_final;
}

/* O problema aqui  fazer com que isto funcione apesar de possveis
   excepes.  Podem ocorrer excepes:

   1.  Ao construir os elos.

   2.  Ao construir os itens nos elos.	Se Item for o mesmo que int, isto
   no acontece, mas ns queremos poder mudar Item para outro tipo
   qualquer, que pode lanar uma excepo ao ser construdo...

   3.  Ao atribuir os itens nos elos, se se reciclarem os elos.	 Mais uma
   vez com Item sendo int, isto no acontece, mas ns queremos poder mudar
   Item  vontade...

   Se alguma coisa falhar, ento considera-se que toda a operao de
   atribuio falhou e portanto a lista implcita deve ficar intacta, tal
   como estava no incio...  Isso implica que no se pode reciclar, pois
   reciclar implica perder os valores dos itens...

   A melhor soluo (proposta por Stroustrup)  simples.  Construir uma
   nova lista por cpia de modelo (o construtor por cpia comporta-se
   bem em caso de erro) e depois trocar os elos dessa cpia com os da
   lista implcita.  Quando o operador terminar, a cpia  destruda
   junto com todos os elos que estavam na lista implcita originalmente.
   Para simplificar a soluo enriqueceu-se a interface da classe com um
   procedimento membro trocaCom() que troca os itens de duas listas.

inline ListaDeInt& ListaInt::operator=(ListaDeInt const& modelo)
{
    assert(cumpreInvariante());
    assert(modelo.cumpreInvariante());


    if(this != &modelo) {
	ListaDeInt copia_da_modelo(modelo);
	
	// Se a copia falhar a troca no chega a ser feita.  Logo, a instncia
	// implcita no  alterada!
	trocaCom(copia_da_modelo);
	}
	// Explique porque  dispensvel o habitual teste if(this != &modelo)...

	return *this;
	}

	Ou mais simples ainda: */

inline ListaDeInt& ListaDeInt::operator=(ListaDeInt copia_de_modelo)
{
    assert(cumpreInvariante());
    assert(copia_de_modelo.cumpreInvariante());

    trocaCom(copia_de_modelo);

    assert(cumpreInvariante());
    assert(copia_de_modelo.cumpreInvariante());

    return *this;
}

inline ListaDeInt::Item const& ListaDeInt::frente() const
{
    assert(cumpreInvariante());
    assert(not estaVazia());

    return elo_inicial->seguinte->item;
}

inline ListaDeInt::Item const& ListaDeInt::tras() const
{
    assert(cumpreInvariante());
    assert(not estaVazia());

    return elo_final->anterior->item;
}

inline int ListaDeInt::comprimento() const
{
    assert(cumpreInvariante());

    return numero_de_itens;
}

inline bool ListaDeInt::estaVazia() const
{
    assert(cumpreInvariante());

    return comprimento() == 0;
}

inline ListaDeInt::IteradorConstante ListaDeInt::primeiro() const
{
    assert(cumpreInvariante());

    return IteradorConstante(*this);
}

inline ListaDeInt::IteradorConstante ListaDeInt::ultimo() const
{
    assert(cumpreInvariante());

    return IteradorConstante(*this, elo_final->anterior);
}

inline ListaDeInt::IteradorConstante ListaDeInt::inicio() const
{
    assert(cumpreInvariante());

    return IteradorConstante(*this, elo_inicial);
}

inline ListaDeInt::IteradorConstante ListaDeInt::fim() const
{
    assert(cumpreInvariante());

    return IteradorConstante(*this, elo_final);
}

inline ListaDeInt::Item& ListaDeInt::frente()
{
    assert(cumpreInvariante());
    assert(not estaVazia());

    return elo_inicial->seguinte->item;
}

inline ListaDeInt::Item& ListaDeInt::tras()
{
    assert(cumpreInvariante());
    assert(not estaVazia());

    return elo_final->anterior->item;
}

/* Os mtodos poeNaFrente(), poeAtras() e insere() so todos
   implementados  custa do mtodo membro privado poe(): */

inline void ListaDeInt::poeNaFrente(Item const& novo_item)
{
    assert(cumpreInvariante());

    poe(elo_inicial, elo_inicial->seguinte, novo_item);

    assert(cumpreInvariante());
}

inline void ListaDeInt::poeAtras(Item const& novo_item)
{
    assert(cumpreInvariante());

    poe(elo_final->anterior, elo_final, novo_item);

    assert(cumpreInvariante());
}

inline void ListaDeInt::insereAntes(Iterador const& iterador,
				    Item const& novo_item)
{
    assert(cumpreInvariante());
    assert(this == iterador.lista());
    assert(iterador != inicio());

    poe(iterador.elo()->anterior, iterador.elo(), novo_item);

    assert(cumpreInvariante());
}

inline void ListaDeInt::insereAntes(IteradorConstante const& iterador,
				    Item const& novo_item)
{
    assert(cumpreInvariante());
    assert(this == iterador.lista());
    assert(iterador != inicio());

    poe(iterador.elo()->anterior, iterador.elo(), novo_item);

    assert(cumpreInvariante());
}

/* Os mtodos tiraDaFrente(), tiraDeTras() e remove() so todos
   implementados  custa do mtodo privado tira(): */

inline void ListaDeInt::tiraDaFrente()
{
    assert(cumpreInvariante());
    assert(not estaVazia());

    tira(elo_inicial->seguinte);

    assert(cumpreInvariante());
}

inline void ListaDeInt::tiraDeTras()
{
    assert(cumpreInvariante());
    assert(not estaVazia());

    tira(elo_final->anterior);

    assert(cumpreInvariante());
}

inline void ListaDeInt::remove(Iterador& iterador)
{
    assert(cumpreInvariante());
    assert(this == iterador.lista());
    assert(iterador != inicio());
    assert(iterador != fim());

    Elo* const elo_a_remover = iterador.elo();

    ++iterador;

    tira(elo_a_remover);

    assert(cumpreInvariante());
}

inline void ListaDeInt::remove(IteradorConstante& iterador)
{
    assert(cumpreInvariante());
    assert(this == iterador.lista());
    assert(iterador != inicio());
    assert(iterador != fim());

    Elo* const elo_a_tirar = iterador.elo();

    ++iterador;

    tira(elo_a_tirar);

    assert(cumpreInvariante());
}

inline void ListaDeInt::remove(Iterador const& iterador)
{
    assert(cumpreInvariante());
    assert(this == iterador.lista());
    assert(iterador != inicio());
    assert(iterador != fim());

    tira(iterador.elo());

    assert(cumpreInvariante());
}

inline void ListaDeInt::remove(IteradorConstante const& iterador)
{
    assert(cumpreInvariante());
    assert(this == iterador.lista());
    assert(iterador != inicio());
    assert(iterador != fim());

    tira(iterador.elo());

    assert(cumpreInvariante());
}

inline void ListaDeInt::transfereDe(ListaDeInt& lista_fonte)
{
    assert(cumpreInvariante());
    assert(lista_fonte.cumpreInvariante());

    /* Se forem a mesma lista transferir ... no fazer nada!  Se
       lista_fonte estiver vazia tambm no  preciso fazer nada: */
    if(this != &lista_fonte and not lista_fonte.estaVazia()) {	
	// Basta trocar ponteiros!
	elo_final->anterior->seguinte = lista_fonte.elo_inicial->seguinte;
	lista_fonte.elo_inicial->seguinte->anterior = elo_final->anterior;
	lista_fonte.elo_final->anterior->seguinte = elo_final;
	elo_final->anterior = lista_fonte.elo_final->anterior;

	lista_fonte.elo_inicial->seguinte = lista_fonte.elo_final;
	lista_fonte.elo_final->anterior = lista_fonte.elo_inicial;
	
	numero_de_itens += lista_fonte.numero_de_itens;
	lista_fonte.numero_de_itens = 0;
    }

    assert(cumpreInvariante());
    assert(lista_fonte.cumpreInvariante());
}

inline void ListaDeInt::trocaCom(ListaDeInt& outra_lista)
{
    assert(cumpreInvariante());
    assert(outra_lista.cumpreInvariante());

    // Este procedimento std::swap() troca o valor de duas variveis do mesmo
    // tipo.  Est declarado em #include <algorithm>:
    std::swap(elo_inicial, outra_lista.elo_inicial);
    std::swap(elo_final, outra_lista.elo_final);
    std::swap(numero_de_itens, outra_lista.numero_de_itens);

    assert(cumpreInvariante());
    assert(outra_lista.cumpreInvariante());
}

inline ListaDeInt& ListaDeInt::operator+=(ListaDeInt lista_fonte)
{
    assert(cumpreInvariante());
    assert(lista_fonte.cumpreInvariante());

    transfereDe(lista_fonte);

    assert(cumpreInvariante());

    return *this;
}

inline ListaDeInt::Iterador ListaDeInt::primeiro()
{
    assert(cumpreInvariante());

    return Iterador(*this);
}

inline ListaDeInt::Iterador ListaDeInt::ultimo()
{
    assert(cumpreInvariante());

    return Iterador(*this, elo_final->anterior);
}

inline ListaDeInt::Iterador ListaDeInt::inicio()
{
    assert(cumpreInvariante());

    return Iterador(*this, elo_inicial);
}

inline ListaDeInt::Iterador ListaDeInt::fim()
{
    assert(cumpreInvariante());

    return Iterador(*this, elo_final);
}

/* Coloca um novo elo, contendo o item novo_item, na cadeia duplamente
   ligada.  O novo elo  colocado entre o elo dado por anterior e o
   elo dado por seguinte. */
inline void ListaDeInt::poe(Elo* const elo_anterior, Elo* const elo_seguinte,
			    Item const& novo_item)
{
    assert(elo_anterior->seguinte == elo_seguinte);
    assert(elo_seguinte->anterior == elo_anterior);

    // Obter novo elo:
    Elo* const novo_elo = new Elo(elo_anterior, elo_seguinte, novo_item);

    ++numero_de_itens;

    elo_anterior->seguinte = novo_elo;
    elo_seguinte->anterior = novo_elo;
}

inline void ListaDeInt::tira(Elo* const elo_a_tirar)
{
    --numero_de_itens;

    elo_a_tirar->anterior->seguinte = elo_a_tirar->seguinte;
    elo_a_tirar->seguinte->anterior = elo_a_tirar->anterior;

    delete elo_a_tirar;
}

inline bool ListaDeInt::cumpreInvariante() const
{
    if(numero_de_itens < 0)
	return false;
    Elo* elo_anterior = elo_inicial;
    Elo* elo = elo_inicial->seguinte;
    int i = 0;
    while(i != numero_de_itens and elo != elo_final) {
	if(elo->anterior != elo_anterior)
	    return false;
	elo_anterior = elo;
	elo = elo->seguinte;
	++i;
    }
    return elo->anterior == elo_anterior and
	elo == elo_final and i == numero_de_itens;
}


inline ListaDeInt::Iterador::Iterador(ListaDeInt& lista_a_associar)
    : lista_associada(&lista_a_associar),
    elo_do_item_referenciado(lista_a_associar.elo_inicial->seguinte)
{
    assert(cumpreInvariante());
}

inline ListaDeInt::Item& ListaDeInt::Iterador::operator*() const
{
    assert(cumpreInvariante());
    assert(*this != lista_associada->inicio());
    assert(*this != lista_associada->fim());

    return elo_do_item_referenciado->item;
}

inline bool
ListaDeInt::Iterador::operator==(Iterador const& outro_iterador) const
{
    assert(cumpreInvariante());
    assert(outro_iterador.cumpreInvariante());
    assert(lista_associada == outro_iterador.lista_associada);

    return elo_do_item_referenciado == outro_iterador.elo_do_item_referenciado;
}

inline bool
ListaDeInt::Iterador::operator!=(Iterador const& outro_iterador) const
{
    assert(cumpreInvariante());
    assert(outro_iterador.cumpreInvariante());
    assert(lista_associada == outro_iterador.lista_associada);

    return not (*this == outro_iterador);
}

inline bool ListaDeInt::Iterador::
operator==(IteradorConstante const& outro_iterador) const
{
    assert(cumpreInvariante());
    assert(outro_iterador.cumpreInvariante());
    assert(lista_associada == outro_iterador.lista_associada);

    return elo_do_item_referenciado == outro_iterador.elo_do_item_referenciado;
}

inline bool ListaDeInt::Iterador::
operator!=(IteradorConstante const& outro_iterador) const
{
    assert(cumpreInvariante());
    assert(outro_iterador.cumpreInvariante());
    assert(lista_associada == outro_iterador.lista_associada);

    return not (*this == outro_iterador);
}

inline ListaDeInt::Iterador& ListaDeInt::Iterador::operator++()
{
    assert(cumpreInvariante());
    assert(*this != lista_associada->fim());

    elo_do_item_referenciado = elo_do_item_referenciado->seguinte;

    assert(cumpreInvariante());

    return *this;
}

inline ListaDeInt::Iterador& ListaDeInt::Iterador::operator--()
{
    assert(cumpreInvariante());
    assert(*this != lista_associada->inicio());

    elo_do_item_referenciado = elo_do_item_referenciado->anterior;

    assert(cumpreInvariante());

    return *this;
}

inline ListaDeInt::Iterador ListaDeInt::Iterador::operator++(int)
{
    assert(cumpreInvariante());

    ListaDeInt::Iterador resultado = *this;
    ++*this;

    assert(cumpreInvariante());

    return resultado;
}

inline ListaDeInt::Iterador ListaDeInt::Iterador::operator--(int)
{
    assert(cumpreInvariante());

    ListaDeInt::Iterador resultado = *this;
    --*this;

    assert(cumpreInvariante());

    return resultado;
}

inline bool ListaDeInt::Iterador::cumpreInvariante() const
{
    /* Em rigor dever-se-ia verificar se o elo pertence  cadeia dos
       elos, pois de outra forma o iterador est corrompido.  Mas a
       verdade  que os elos podem estar em estados invlidos... */
    return true /* simplificao grosseira. */;
}

inline ListaDeInt::Iterador::Iterador(ListaDeInt& lista_a_associar,
				      Elo* const elo_do_item_a_referenciar)
    : lista_associada(&lista_a_associar),
    elo_do_item_referenciado(elo_do_item_a_referenciar)
{
    assert(cumpreInvariante());
}

inline ListaDeInt const* ListaDeInt::Iterador::lista() const
{
    assert(cumpreInvariante());

    return lista_associada;
}

inline ListaDeInt::Elo* ListaDeInt::Iterador::elo() const
{
    assert(cumpreInvariante());

    return elo_do_item_referenciado;
}

inline ListaDeInt::IteradorConstante::
IteradorConstante(ListaDeInt const& lista_a_associar)
    : lista_associada(&lista_a_associar),
    elo_do_item_referenciado(lista_a_associar.elo_inicial->seguinte)
{
    assert(cumpreInvariante());
}

inline ListaDeInt::IteradorConstante::
IteradorConstante(Iterador const& outro_iterador)
    : lista_associada(outro_iterador.lista()),
    elo_do_item_referenciado(outro_iterador.elo())
{
    assert(cumpreInvariante());
}

inline ListaDeInt::Item const& ListaDeInt::IteradorConstante::
operator*() const
{
    assert(cumpreInvariante());
    assert(*this != lista_associada->inicio());
    assert(*this != lista_associada->fim());

    return elo_do_item_referenciado->item;
}

inline bool ListaDeInt::IteradorConstante::
operator==(IteradorConstante const& outro_iterador) const
{
    assert(cumpreInvariante());
    assert(outro_iterador.cumpreInvariante());
    assert(lista_associada == outro_iterador.lista_associada);

    return elo_do_item_referenciado == outro_iterador.elo_do_item_referenciado;
}

inline bool ListaDeInt::IteradorConstante::
operator!=(IteradorConstante const& outro_iterador) const
{
    return not (*this == outro_iterador);
}

inline ListaDeInt::IteradorConstante& ListaDeInt::IteradorConstante::
operator++()
{
    assert(cumpreInvariante());
    assert(*this != lista_associada->fim());

    elo_do_item_referenciado = elo_do_item_referenciado->seguinte;

    assert(cumpreInvariante());

    return *this;
}

inline ListaDeInt::IteradorConstante& ListaDeInt::IteradorConstante::
operator--()
{
    assert(cumpreInvariante());
    assert(*this != lista_associada->inicio());

    elo_do_item_referenciado = elo_do_item_referenciado->anterior;

    assert(cumpreInvariante());

    return *this;
}

inline ListaDeInt::IteradorConstante ListaDeInt::IteradorConstante::
operator++(int)
{
    assert(cumpreInvariante());

    ListaDeInt::IteradorConstante resultado = *this;
    ++*this;

    assert(cumpreInvariante());

    return resultado;
}

inline ListaDeInt::IteradorConstante
ListaDeInt::IteradorConstante::operator--(int) {
    assert(cumpreInvariante());

    ListaDeInt::IteradorConstante resultado = *this;
    --*this;

    assert(cumpreInvariante());

    return resultado;
}

inline bool ListaDeInt::IteradorConstante::cumpreInvariante() const
{
    /* Em rigor dever-se-ia verificar se o elo pertence  cadeia dos
       elos, pois de outra forma o iterador est corrompido.  Mas a verdade 
       que os elos podem estar em estados invlidos... */
    return true /* simplificao grosseira. */;
}

inline ListaDeInt::IteradorConstante::
IteradorConstante(ListaDeInt const& lista_a_associar,
		  Elo* const elo_do_item_a_referenciar)
    : lista_associada(&lista_a_associar),
    elo_do_item_referenciado(elo_do_item_a_referenciar)
{
    assert(cumpreInvariante());
}

inline ListaDeInt const* ListaDeInt::IteradorConstante::lista() const
{
    assert(cumpreInvariante());

    return lista_associada;
}

inline ListaDeInt::Elo* ListaDeInt::IteradorConstante::elo() const
{
    assert(cumpreInvariante());

    return elo_do_item_referenciado;
}
