#ifndef TERMOS_H
#define TERMOS_H

#include "termo.H"

#include <string>
#include <memory>

/** Representa adies numa expresso.
    @invariant termo_esquerdo.get() <> 0 e termo_direito.get() <> 0 e nenhum 
        dos termos corresponde directa ou indirectamente a esta mesma 
	adio. */
class Adicao : public Termo {
public:
    /** Constri uma nova adio dados os dois termos correspondentes aos seus
	operandos.  Os termos passam a ser posse exclusiva da adio.
	@pre termo_esquerdo.get() <> 0 e termo_direito.get() <> 0.
	@post valor() = termo_esquerdo->valor() + termo_direito->valor(). */
    Adicao(std::auto_ptr<Termo> termo_esquerdo, 
	   std::auto_ptr<Termo> termo_direito);

    /** Destrutor polimrfico marca a classe como polimrfica.
	@pre V. */
    virtual ~Adicao() {}

    /** Devolve o valor do termo.
	@pre V.
	@post valor = valor do termo. */
    virtual double valor() const;

    /** Devolve o termo representado numa cadeia de caracteres.  O termo  
	representado como uma expresso com todos os seus parnteses.
	@pre V.
	@post comoCadeia = expresso com todos os parnteses correspondente ao
	      termo. */
    virtual std::string const comoCadeia() const;

private:
    std::auto_ptr<Termo const> termo_esquerdo;
    std::auto_ptr<Termo const> termo_direito;

    /** Indica se a condio invariante da classe se verifica.
	@pre V.
	@post cumpreInvariante = termo_esquerdo.get() <> 0 and 
	    termo_direito.get() <> 0. */
    bool cumpreInvariante() const;
};

/** Representa produtos numa expresso.
    @invariant termo_esquerdo.get() <> 0 e termo_direito.get() <> 0 e nenhum 
        dos termos corresponde directa ou indirectamente a este mesmo 
	produto. */
class Produto : public Termo {
public:
    /** Constri um novo produto dados os dois termos correspondentes aos seus
	operandos.  Os termos passam a ser posse exclusiva do produto.
	@pre termo_esquerdo.get() <> 0 e termo_direito.get() <> 0.
	@post valor() = termo_esquerdo->valor() * termo_direito->valor(). */
    Produto(std::auto_ptr<Termo> termo_esquerdo, 
	    std::auto_ptr<Termo> termo_direito);

    /** Destrutor polimrfico marca a classe como polimrfica.
	@pre V. */
    virtual ~Produto() {}

    /** Devolve o valor do termo.
	@pre V.
	@post valor = valor do termo. */
    virtual double valor() const;

    /** Devolve o termo representado numa cadeia de caracteres.  O termo  
	representado como uma expresso com todos os seus parnteses.
	@pre V.
	@post comoCadeia = expresso com todos os parnteses correspondente ao
	      termo. */
    virtual std::string const comoCadeia() const;

private:
    std::auto_ptr<Termo const> termo_esquerdo;
    std::auto_ptr<Termo const> termo_direito;

    /** Indica se a condio invariante da classe se verifica.
	@pre V.
	@post cumpreInvariante = termo_esquerdo.get() <> 0 and 
	    termo_direito.get() <> 0. */
    bool cumpreInvariante() const;
};

/** Representa potncias numa expresso.
    @invariant termo_esquerdo.get() <> 0 e termo_direito.get() <> 0 e nenhum 
        dos termos corresponde directa ou indirectamente a esta mesma 
	potncia. */
class Potencia : public Termo {
public:
    /** Constri uma nova potncia dados os dois termos correspondentes aos seus
	operandos.  Os termos passam a ser posse exclusiva da potncia.
	@pre termo_esquerdo.get() <> 0 e termo_direito.get() <> 0.
	@post valor() = termo_esquerdo->valor() ^ termo_direito->valor(). */
    Potencia(std::auto_ptr<Termo> termo_esquerdo, 
	     std::auto_ptr<Termo> termo_direito);

    /** Destrutor polimrfico marca a classe como polimrfica.
	@pre V. */
    virtual ~Potencia() {}

    /** Devolve o valor do termo.
	@pre V.
	@post valor = valor do termo. */
    virtual double valor() const;

    /** Devolve o termo representado numa cadeia de caracteres.  O termo  
	representado como uma expresso com todos os seus parnteses.
	@pre V.
	@post comoCadeia = expresso com todos os parnteses correspondente ao
	      termo. */
    virtual std::string const comoCadeia() const;

private:
    std::auto_ptr<Termo const> termo_esquerdo;
    std::auto_ptr<Termo const> termo_direito;

    /** Indica se a condio invariante da classe se verifica.
	@pre V.
	@post cumpreInvariante = termo_esquerdo.get() <> 0 and 
	    termo_direito.get() <> 0. */
    bool cumpreInvariante() const;
};

class Valor : public Termo {
public:
    Valor(double valor);
    virtual ~Valor() {}

    virtual double valor() const;
    virtual std::string const comoCadeia() const;

private:
    double valor_;

    bool cumpreInvariante() const;
};

#include "termos_impl.H"

#endif // TERMO_H
