#ifndef QUADRO_H
#define QUADRO_H

#include <iostream>
#include <vector>

#include "cor.H"
#include "posicao.H"
#include "dimensao.H"

/** Representa um quadro onde se pode desenhar.  O quadro est
    dividido em rectngulos organizados em linhas e colunas.  Cada
    rectngulo chama-se um pxel.  A linha do topo  a linha 0 (zero).
    A coluna da esquerda  a coluna 0 (zero).  Os pxeis podem ser
    pintados de branco ou de preto.  A razo entre a largura e altura
    dos pxeis chama-se aspecto.   possvel escrever um quadro num
    canal, usualmente ligado ao ecr.  Nesse caso cada pxel 
    representado por um caracter, sendo ' ' usado para a cor preta e
    '' usado para a cor branca.
    @invariant 0 < dimensao_.numeroDeLinhas() e 
               0 < dimensao_.numeroDeColunas() e
               0.0 < aspecto_ e
               pixeis.size() = dimensao_.numeroDeLinhas() e 
               (Q i : 0 <= i < pixeis.size() : 
                pixeis[i] = dimensao_.numeroDeColunas()). */
class Quadro {
  public:
    // Usa-se preto e branco:
    typedef PretoEBranco Cor;

    /** Constri um quadro com dimenso dada (24x80 por omisso) e com
        fundo da cor fundo (preto por omisso).  Os pxeis vlidos tm
        posies entre 0 e dimenso.numeroDeLinhas() - 1 (para a
        linha) e entre 0 e dimenso.numeroDeColunas() - 1 (para a
        coluna).  A razo entre a largura e a altura dos pxeis  dada
        por aspecto.
        @pre 0 < dimensao.numeroDeLinhas() e 0 < dimensao.numeroDeColunas() e
             0.0 <= aspecto.
        @post dimensao() = dimensao e fundo() = fundo e aspecto() = aspecto. */
    Quadro(Dimensao const& dimensao = Dimensao(24, 80), 
           Cor const fundo = preto,
           double const aspecto = 1.0);

    /** Devolve a dimensao do quadro.
        @pre V.
        @post dimensao = dimenso do quadro. */
    Dimensao dimensao() const;

    /** Devolve a razo largura/altura (i.e., o aspecto) dos pxeis
        do quadro.
        @pre V.
        @post aspecto = aspecto dos pxeis no quadro. */
    double aspecto() const;

    /** Devolve a cor do fundo.
        @pre V.
        @post fundo = cor do fundo do quadro. */
    Cor fundo() const;

    /** Indica se uma dada posio est dentro do quadro.
        @pre V.
        @post inclui = (0 <= posicao.linha() e
                        posicao.linha() < dimensao().numeroDeLinhas() e
                        0 <= posicao.coluna() e
                        posicao.coluna() < dimensao().numeroDeColunas()). */
    bool inclui(Posicao const& posicao) const;

    /** Devolve a cor do pixel na posio dada.  Se a posio estiver
        fora do quadro devolve a cor do fundo. 
        @pre V.
        @post (cor = fundo() e inclui(posicao)) ou 
              (cor = cor do pxel em posicao e inclui(posicao)). */
    Cor cor(Posicao const& posicao) const;

    /** Escreve o quadro no canal saida (que se presume ligado a um
        ecr em modo texto), com converso para os caracteres ' ' e
        '' (preto e branco respectivamente).
        @pre V.
        @post saida.fail() ou (saida.fail() e saida contm uma 
              representao do quadro no modo texto). */
    void mostraEm(std::ostream& saida) const;

    /** Pinta o pxel numa dada posio com uma dada cor.
        @pre V.
        @post inclui(posicao) ou cor(posicao) = cor. */
    void pinta(Posicao const& posicao, Cor const cor);

    /** Pinta o pixel numa dada posio com um cor contrastante com o
        fundo do quadro.
        @pre V.
        @post inclui(posicao) ou cor(posicao) = contrasta(fundo()). */
    void pinta(Posicao const& posicao);

    /** "Despinta" o pixel numa dada posio pintando-o com a cor do
        fundo.
        @pre V.
        @post cor(posicao) = fundo(). */
    void limpa(Posicao const& posicao);

    /** Pinta todo o quadro com a cor do fundo.
        @pre V.
        @post (Q p : : cor(p) = fundo()). */
    void apaga();

  private:
    // Dimenso do quadro.
    Dimensao dimensao_;         

    // Cor de fundo do quadro.
    Cor fundo_;

    // Razo largura/altura dos pxeis do quadro.
    double aspecto_;

    // Vector bi-dimensional de pxeis.
    std::vector<std::vector<Cor> > pixeis;

    /** Indica se a condio invariante da classe se verifica.
        @pre V.
        @post cumpreInvariante = (
               0 < dimensao_.numeroDeLinhas() e 
               0 < dimensao_.numeroDeColunas() e
               0.0 < aspecto_ e
               pixeis.size() = dimensao_.numeroDeLinhas() e 
               (Q i : 0 <= i < pixeis.size() : 
                pixeis[i] = dimensao_.numeroDeColunas()). */
    bool cumpreInvariante() const;
};

#include "quadro_impl.H"

#endif // QUADRO_H

